"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTransforms = exports.compile = void 0;
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _content_management_services_schemas = require("./content_management_services_schemas");
var _utils = require("./utils");
var _object_transform = require("./object_transform");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const serviceObjectPaths = ['get.in.options', 'get.out.result', 'bulkGet.in.options', 'bulkGet.out.result', 'create.in.options', 'create.in.data', 'create.out.result', 'update.in.options', 'update.in.data', 'update.out.result', 'delete.in.options', 'delete.out.result', 'search.in.options', 'search.out.result', 'mSearch.out.result'];
const validateServiceDefinitions = definitions => {
  if (definitions === null || Array.isArray(definitions) || typeof definitions !== 'object') {
    throw new Error('Invalid service definition. Must be an object.');
  }
  if (Object.keys(definitions).length === 0) {
    throw new Error('At least one version must be defined.');
  }
  Object.entries(definitions).forEach(([version, definition]) => {
    const {
      result: isVersionValid
    } = (0, _utils.validateVersion)(version);
    if (!isVersionValid) {
      throw new Error(`Invalid version [${version}]. Must be an integer.`);
    }
    const error = (0, _utils.validateObj)(definition, _content_management_services_schemas.serviceDefinitionSchema);
    if (error !== null) {
      throw new Error(`Invalid services definition. [${error}]`);
    }
  });
};

/**
 * Convert a versionned service definition to a flattened service definition
 * where _each object_ is versioned (at the leaf).
 *
 * @example
 *
 * ```ts
 * From this
 * {
 *   1: {
 *     get: {
 *       in: {
 *         options: { up: () => {} } // 1
 *       }
 *     },
 *     ...
 *   },
 *   2: {
 *     get: {
 *       in: {
 *         options: { up: () => {} } // 2
 *       }
 *     },
 *   }
 * }
 *
 * To this
 *
 * {
 *   'get.in.options': { // Flattend path
 *      1: { up: () => {} }, // 1
 *      2: { up: () => {} }  // 2
 *    }
 * }
 * ```
 */
const compile = definitions => {
  validateServiceDefinitions(definitions);
  const flattened = {};
  Object.entries(definitions).forEach(([version, definition]) => {
    serviceObjectPaths.forEach(path => {
      var _get, _get2;
      const versionableObject = (_get = (0, _lodash.get)(definition, path)) !== null && _get !== void 0 ? _get : {};
      const objectMigrationDefinition = {
        ...((_get2 = (0, _lodash.get)(flattened, path)) !== null && _get2 !== void 0 ? _get2 : {}),
        [version]: versionableObject
      };
      flattened[path] = objectMigrationDefinition;
    });
  });
  return flattened;
};
exports.compile = compile;
const getDefaultTransforms = () => ({
  up: input => input,
  down: input => input,
  validate: () => null
});
const getDefaultServiceTransforms = () => ({
  get: {
    in: {
      options: getDefaultTransforms()
    },
    out: {
      result: getDefaultTransforms()
    }
  },
  bulkGet: {
    in: {
      options: getDefaultTransforms()
    },
    out: {
      result: getDefaultTransforms()
    }
  },
  create: {
    in: {
      options: getDefaultTransforms(),
      data: getDefaultTransforms()
    },
    out: {
      result: getDefaultTransforms()
    }
  },
  update: {
    in: {
      options: getDefaultTransforms(),
      data: getDefaultTransforms()
    },
    out: {
      result: getDefaultTransforms()
    }
  },
  delete: {
    in: {
      options: getDefaultTransforms()
    },
    out: {
      result: getDefaultTransforms()
    }
  },
  search: {
    in: {
      options: getDefaultTransforms()
    },
    out: {
      result: getDefaultTransforms()
    }
  },
  mSearch: {
    out: {
      result: getDefaultTransforms()
    }
  }
});
const getTransforms = (definitions, requestVersion, _compiled) => {
  // Compile the definition into a flattened object with ObjectMigrationDefinition
  const compiled = _compiled !== null && _compiled !== void 0 ? _compiled : compile(definitions);

  // Initiate transform for specific request version
  const transformsForRequest = getDefaultServiceTransforms();
  Object.entries(compiled).forEach(([path, objectMigrationDefinition]) => {
    const objectTransforms = (0, _object_transform.initTransform)(requestVersion)(objectMigrationDefinition);
    (0, _saferLodashSet.set)(transformsForRequest, path, objectTransforms);
  });
  return transformsForRequest;
};
exports.getTransforms = getTransforms;