"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MvtVectorLayer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _i18n = require("@kbn/i18n");
var _esQuery = require("@kbn/es-query");
var _vector_style = require("../../../styles/vector/vector_style");
var _elasticsearch_util = require("../../../../../common/elasticsearch_util");
var _constants = require("../../../../../common/constants");
var _vector_layer = require("../vector_layer");
var _mvt_source_data = require("./mvt_source_data");
var _pluck_style_meta = require("./pluck_style_meta");
var _tile_meta_feature_utils = require("../../../util/tile_meta_feature_utils");
var _bounds_data = require("../bounds_data");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_RESULT_WINDOW_DATA_REQUEST_ID = 'maxResultWindow';
class MvtVectorLayer extends _vector_layer.AbstractVectorLayer {
  static createDescriptor(descriptor, mapColors) {
    const layerDescriptor = super.createDescriptor(descriptor, mapColors);
    layerDescriptor.type = _constants.LAYER_TYPE.MVT_VECTOR;
    if (!layerDescriptor.style) {
      const styleProperties = _vector_style.VectorStyle.createDefaultStyleProperties(mapColors ? mapColors : []);
      layerDescriptor.style = _vector_style.VectorStyle.createDescriptor(styleProperties);
    }
    return layerDescriptor;
  }
  constructor(args) {
    super(args);
    (0, _defineProperty2.default)(this, "_source", void 0);
    this._source = args.source;
  }
  isLayerLoading(zoom) {
    if (!this.isVisible() || !this.showAtZoomLevel(zoom)) {
      return false;
    }
    const isSourceLoading = super.isLayerLoading(zoom);
    return isSourceLoading ? true : this._isLoadingJoins();
  }
  _isTiled() {
    // Uses tiled maplibre source 'vector'
    return true;
  }
  async getBounds(getDataRequestContext) {
    // Add filter to narrow bounds to features with matching join keys
    let joinKeyFilter;
    if (this.getSource().isESSource()) {
      const {
        join,
        joinPropertiesMap
      } = this._getJoinResults();
      if (join && joinPropertiesMap) {
        const indexPattern = await this.getSource().getIndexPattern();
        const joinField = (0, _elasticsearch_util.getField)(indexPattern, join.getLeftField().getName());
        joinKeyFilter = (0, _esQuery.buildPhrasesFilter)(joinField, Array.from(joinPropertiesMap.keys()), indexPattern);
      }
    }
    const syncContext = getDataRequestContext(this.getId());
    return (0, _bounds_data.syncBoundsData)({
      layerId: this.getId(),
      syncContext: {
        ...syncContext,
        dataFilters: {
          ...syncContext.dataFilters,
          joinKeyFilter
        }
      },
      source: this.getSource(),
      sourceQuery: this.getQuery()
    });
  }
  getFeatureId(feature) {
    var _feature$properties, _feature$properties2;
    if (!this.getSource().isESSource()) {
      return feature.id;
    }
    return this.getSource().getType() === _constants.SOURCE_TYPES.ES_SEARCH ? (_feature$properties = feature.properties) === null || _feature$properties === void 0 ? void 0 : _feature$properties._id : (_feature$properties2 = feature.properties) === null || _feature$properties2 === void 0 ? void 0 : _feature$properties2._key;
  }
  getLayerIcon(isTocIcon) {
    if (!this.getSource().isESSource()) {
      // Only ES-sources can have a special meta-tile, not 3rd party vector tile sources
      return {
        icon: this.getCurrentStyle().getIcon(false),
        tooltipContent: null,
        areResultsTrimmed: false
      };
    }

    //
    // TODO ES MVT specific - move to es_tiled_vector_layer implementation
    //
    if (this.getSource().getType() === _constants.SOURCE_TYPES.ES_GEO_GRID) {
      const {
        docCount
      } = (0, _tile_meta_feature_utils.getAggsMeta)(this._getTileMetaFeatures());
      return docCount === 0 ? _vector_layer.NO_RESULTS_ICON_AND_TOOLTIPCONTENT : {
        icon: this.getCurrentStyle().getIcon(false),
        tooltipContent: null,
        areResultsTrimmed: false
      };
    }
    const maxResultWindow = this._getMaxResultWindow();
    if (maxResultWindow === undefined) {
      return {
        icon: this.getCurrentStyle().getIcon(false),
        tooltipContent: null,
        areResultsTrimmed: false
      };
    }
    const {
      totalFeaturesCount,
      tilesWithFeatures,
      tilesWithTrimmedResults
    } = (0, _tile_meta_feature_utils.getHitsMeta)(this._getTileMetaFeatures(), maxResultWindow);
    if (totalFeaturesCount === 0) {
      return _vector_layer.NO_RESULTS_ICON_AND_TOOLTIPCONTENT;
    }
    const areResultsTrimmed = tilesWithTrimmedResults > 0;

    // Documents may be counted multiple times if geometry crosses tile boundaries.
    const canMultiCountShapes = !this.getStyle().getIsPointsOnly() && totalFeaturesCount > 1 && tilesWithFeatures > 1;
    const countPrefix = canMultiCountShapes ? '~' : '';
    const countMsg = areResultsTrimmed ? _i18n.i18n.translate('xpack.maps.tiles.resultsTrimmedMsg', {
      defaultMessage: `Results limited to {countPrefix}{count} documents.`,
      values: {
        count: totalFeaturesCount.toLocaleString(),
        countPrefix
      }
    }) : _i18n.i18n.translate('xpack.maps.tiles.resultsCompleteMsg', {
      defaultMessage: `Found {countPrefix}{count} documents.`,
      values: {
        count: totalFeaturesCount.toLocaleString(),
        countPrefix
      }
    });
    const tooltipContent = canMultiCountShapes ? countMsg + _i18n.i18n.translate('xpack.maps.tiles.shapeCountMsg', {
      defaultMessage: ' This count is approximate.'
    }) : countMsg;
    return {
      icon: this.getCurrentStyle().getIcon(isTocIcon && areResultsTrimmed),
      tooltipContent,
      areResultsTrimmed
    };
  }
  _getMaxResultWindow() {
    const dataRequest = this.getDataRequest(MAX_RESULT_WINDOW_DATA_REQUEST_ID);
    if (!dataRequest) {
      return;
    }
    const data = dataRequest.getData();
    return data ? data.maxResultWindow : undefined;
  }
  async _syncMaxResultWindow({
    startLoading,
    stopLoading
  }) {
    const prevDataRequest = this.getDataRequest(MAX_RESULT_WINDOW_DATA_REQUEST_ID);
    if (prevDataRequest) {
      return;
    }
    const requestToken = Symbol(`${this.getId()}-${MAX_RESULT_WINDOW_DATA_REQUEST_ID}`);
    startLoading(MAX_RESULT_WINDOW_DATA_REQUEST_ID, requestToken);
    const maxResultWindow = await this.getSource().getMaxResultWindow();
    stopLoading(MAX_RESULT_WINDOW_DATA_REQUEST_ID, requestToken, {
      maxResultWindow
    });
  }
  async syncData(syncContext) {
    try {
      if (this.getSource().getType() === _constants.SOURCE_TYPES.ES_SEARCH) {
        await this._syncMaxResultWindow(syncContext);
      }
      await this._syncSourceStyleMeta(syncContext, this.getSource(), this.getCurrentStyle());
      await this._syncSourceFormatters(syncContext, this.getSource(), this.getCurrentStyle());
      await this._syncSupportsFeatureEditing({
        syncContext,
        source: this.getSource()
      });
      let maxLineWidth = 0;
      const lineWidth = this.getCurrentStyle().getAllStyleProperties().find(styleProperty => {
        return styleProperty.getStyleName() === _constants.VECTOR_STYLES.LINE_WIDTH;
      });
      if (lineWidth) {
        if (!lineWidth.isDynamic() && lineWidth.isComplete()) {
          maxLineWidth = lineWidth.getOptions().size;
        } else if (lineWidth.isDynamic() && lineWidth.isComplete()) {
          maxLineWidth = lineWidth.getOptions().maxSize;
        }
      }
      const buffer = Math.ceil(3.5 * maxLineWidth);
      await (0, _mvt_source_data.syncMvtSourceData)({
        buffer,
        hasLabels: this.getCurrentStyle().hasLabels(),
        layerId: this.getId(),
        layerName: await this.getDisplayName(),
        prevDataRequest: this.getSourceDataRequest(),
        requestMeta: await this._getVectorSourceRequestMeta(syncContext.isForceRefresh, syncContext.dataFilters, this.getSource(), this.getCurrentStyle(), syncContext.isFeatureEditorOpenForLayer),
        source: this.getSource(),
        syncContext
      });
      if (this.hasJoins()) {
        await this._syncJoins(syncContext, this.getCurrentStyle());
      }
    } catch (error) {
      // Error used to stop execution flow. Error state stored in data request and displayed to user in layer legend.
    }
  }
  _syncSourceBindingWithMb(mbMap) {
    const mbSource = mbMap.getSource(this.getMbSourceId());
    if (mbSource) {
      return;
    }
    const sourceDataRequest = this.getSourceDataRequest();
    if (!sourceDataRequest) {
      // this is possible if the layer was invisible at startup.
      // the actions will not perform any data=syncing as an optimization when a layer is invisible
      // when turning the layer back into visible, it's possible the url had not been resolved yet.
      return;
    }
    const sourceData = sourceDataRequest.getData();
    if (!sourceData) {
      return;
    }
    const mbSourceId = this.getMbSourceId();
    mbMap.addSource(mbSourceId, {
      type: 'vector',
      tiles: [sourceData.tileUrl],
      minzoom: sourceData.tileMinZoom,
      maxzoom: sourceData.tileMaxZoom,
      promoteId: this._getSourcePromoteId()
    });
  }
  getMbLayerIds() {
    return [...super.getMbLayerIds(), this._getMbTooManyFeaturesLayerId()];
  }
  ownsMbSourceId(mbSourceId) {
    return this.getMbSourceId() === mbSourceId;
  }
  _getJoinResults() {
    const joins = this.getValidJoins();
    if (!joins || !joins.length) {
      return {};
    }
    const join = joins[0];
    const joinDataRequest = this.getDataRequest(join.getSourceDataRequestId());
    return {
      join,
      joinPropertiesMap: joinDataRequest === null || joinDataRequest === void 0 ? void 0 : joinDataRequest.getData(),
      joinRequestMeta: joinDataRequest === null || joinDataRequest === void 0 ? void 0 : joinDataRequest.getMeta()
    };
  }
  _getMbTooManyFeaturesLayerId() {
    return this.makeMbLayerId('toomanyfeatures');
  }
  _getJoinFilterExpression() {
    const {
      join,
      joinPropertiesMap
    } = this._getJoinResults();
    if (!join) {
      return undefined;
    }

    // When there are no join results, return a filter that hides all features
    // work around for 'match' with empty array not filtering out features
    // This filter always returns false because features will never have `__kbn_never_prop__` property
    const hideAllFilter = ['has', '__kbn_never_prop__'];
    if (!joinPropertiesMap) {
      return hideAllFilter;
    }
    const joinKeys = Array.from(joinPropertiesMap.keys());
    return joinKeys.length ?
    // Unable to check FEATURE_VISIBLE_PROPERTY_NAME flag since filter expressions do not support feature-state
    // Instead, remove unjoined source features by filtering out features without matching join keys
    ['match', ['get', join.getLeftField().getName()], joinKeys, true,
    // match value
    false // fallback - value with no match
    ] : hideAllFilter;
  }
  _syncFeatureState(mbMap) {
    const {
      joinPropertiesMap,
      joinRequestMeta
    } = this._getJoinResults();
    if (!joinPropertiesMap) {
      return;
    }
    const [firstKey] = joinPropertiesMap.keys();
    const firstKeyFeatureState = mbMap.getFeatureState({
      source: this.getMbSourceId(),
      sourceLayer: this._source.getTileSourceLayer(),
      id: firstKey
    });
    const joinRequestStopTime = joinRequestMeta === null || joinRequestMeta === void 0 ? void 0 : joinRequestMeta.requestStopTime;
    if ((firstKeyFeatureState === null || firstKeyFeatureState === void 0 ? void 0 : firstKeyFeatureState.requestStopTime) === joinRequestStopTime) {
      // Do not update feature state when it already contains current join results
      return;
    }

    // Clear existing feature state
    mbMap.removeFeatureState({
      source: this.getMbSourceId(),
      sourceLayer: this._source.getTileSourceLayer()
      // by omitting 'id' argument, all feature state is cleared for source
    });

    // Set feature state for join results
    // reusing featureIdentifier to avoid creating new object in tight loops
    const featureIdentifier = {
      source: this.getMbSourceId(),
      sourceLayer: this._source.getTileSourceLayer(),
      id: undefined
    };
    joinPropertiesMap.forEach((value, key) => {
      featureIdentifier.id = key;
      mbMap.setFeatureState(featureIdentifier, {
        ...value,
        requestStopTime: joinRequestStopTime
      });
    });
  }
  _syncStylePropertiesWithMb(mbMap) {
    // @ts-ignore
    const mbSource = mbMap.getSource(this.getMbSourceId());
    if (!mbSource) {
      return;
    }
    const sourceDataRequest = this.getSourceDataRequest();
    if (!sourceDataRequest) {
      return;
    }
    const sourceData = sourceDataRequest.getData();
    if (!sourceData || sourceData.tileSourceLayer === '') {
      return;
    }
    this._setMbLabelProperties(mbMap, sourceData.tileSourceLayer);
    this._setMbPointsProperties(mbMap, sourceData.tileSourceLayer);
    this._setMbLinePolygonProperties(mbMap, sourceData.tileSourceLayer);
    this._syncTooManyFeaturesProperties(mbMap);
  }

  // TODO ES MVT specific - move to es_tiled_vector_layer implementation
  _syncTooManyFeaturesProperties(mbMap) {
    if (this.getSource().getType() !== _constants.SOURCE_TYPES.ES_SEARCH) {
      return;
    }
    const maxResultWindow = this._getMaxResultWindow();
    if (maxResultWindow === undefined) {
      return;
    }
    const tooManyFeaturesLayerId = this._getMbTooManyFeaturesLayerId();
    if (!mbMap.getLayer(tooManyFeaturesLayerId)) {
      const mbTooManyFeaturesLayer = {
        id: tooManyFeaturesLayerId,
        type: 'line',
        source: this.getId(),
        paint: {}
      };
      mbTooManyFeaturesLayer['source-layer'] = _tile_meta_feature_utils.ES_MVT_META_LAYER_NAME;
      mbMap.addLayer(mbTooManyFeaturesLayer);
      mbMap.setFilter(tooManyFeaturesLayerId, ['all', ['==', ['get', _tile_meta_feature_utils.ES_MVT_HITS_TOTAL_RELATION], 'gte'], ['>=', ['get', _tile_meta_feature_utils.ES_MVT_HITS_TOTAL_VALUE], maxResultWindow + 1]]);
      mbMap.setPaintProperty(tooManyFeaturesLayerId, 'line-color', this.getCurrentStyle().getPrimaryColor());
      mbMap.setPaintProperty(tooManyFeaturesLayerId, 'line-width', 3);
      mbMap.setPaintProperty(tooManyFeaturesLayerId, 'line-dasharray', [2, 1]);
      mbMap.setPaintProperty(tooManyFeaturesLayerId, 'line-opacity', this.getAlpha());
    }
    this.syncVisibilityWithMb(mbMap, tooManyFeaturesLayerId);
    mbMap.setLayerZoomRange(tooManyFeaturesLayerId, this.getMinZoom(), this.getMaxZoom());
  }
  _getSourcePromoteId() {
    const {
      join
    } = this._getJoinResults();
    return join ? {
      [this._source.getTileSourceLayer()]: join.getLeftField().getName()
    } : undefined;
  }

  // Maplibre does not expose API for updating source attributes.
  // Must remove/add vector source to update source attributes.
  // _requiresPrevSourceCleanup returns true when vector source needs to be removed so it can be re-added with updated attributes
  _requiresPrevSourceCleanup(mbMap) {
    var _mbTileSource$tiles;
    const mbSource = mbMap.getSource(this.getMbSourceId());
    if (!mbSource) {
      return false;
    }
    if (!('tiles' in mbSource)) {
      // Expected source is not compatible, so remove.
      return true;
    }
    const mbTileSource = mbSource;
    const sourceDataRequest = this.getSourceDataRequest();
    if (!sourceDataRequest) {
      return false;
    }
    const sourceData = sourceDataRequest.getData();
    if (!sourceData) {
      return false;
    }
    const isSourceDifferent = ((_mbTileSource$tiles = mbTileSource.tiles) === null || _mbTileSource$tiles === void 0 ? void 0 : _mbTileSource$tiles[0]) !== sourceData.tileUrl || mbTileSource.minzoom !== sourceData.tileMinZoom || mbTileSource.maxzoom !== sourceData.tileMaxZoom || !_lodash.default.isEqual(mbTileSource.promoteId, this._getSourcePromoteId());
    if (isSourceDifferent) {
      return true;
    }
    const layerIds = this.getMbLayerIds();
    for (let i = 0; i < layerIds.length; i++) {
      const mbLayer = mbMap.getLayer(layerIds[i]);
      // The mapbox type in the spec is specified with `source-layer`
      // but the programmable JS-object uses camelcase `sourceLayer`
      if (mbLayer && mbLayer.sourceLayer !== sourceData.tileSourceLayer && mbLayer.sourceLayer !== _tile_meta_feature_utils.ES_MVT_META_LAYER_NAME) {
        // If the source-pointer of one of the layers is stale, they will all be stale.
        // In this case, all the mb-layers need to be removed and re-added.
        return true;
      }
    }
    return false;
  }
  syncLayerWithMB(mbMap) {
    this._removeStaleMbSourcesAndLayers(mbMap);
    this._syncSourceBindingWithMb(mbMap);
    this._syncFeatureState(mbMap);
    this._syncStylePropertiesWithMb(mbMap);
  }
  getMinZoom() {
    // higher resolution vector tiles cannot be displayed at lower-res
    return Math.max(this._source.getMinZoom(), super.getMinZoom());
  }
  getFeatureById(id) {
    return null;
  }
  async getStyleMetaDescriptorFromLocalFeatures() {
    const {
      joinPropertiesMap
    } = this._getJoinResults();
    return await (0, _pluck_style_meta.pluckStyleMeta)(this._getTileMetaFeatures(), joinPropertiesMap, await this.getSource().getSupportedShapeTypes(), this.getCurrentStyle().getDynamicPropertiesArray());
  }
}
exports.MvtVectorLayer = MvtVectorLayer;