"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LensStorage = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _contentManagementUtils = require("@kbn/content-management-utils");
var _content_management = require("../../common/content_management");
var _cm_services = require("../../common/content_management/cm_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const searchArgsToSOFindOptions = args => {
  const {
    query,
    contentTypeId,
    options
  } = args;
  return {
    type: contentTypeId,
    searchFields: ['title^3', 'description'],
    fields: ['description', 'title'],
    search: query.text,
    perPage: query.limit,
    page: query.cursor ? +query.cursor : undefined,
    defaultSearchOperator: 'AND',
    ...options,
    ...(0, _contentManagementUtils.tagsToFindOptions)(query.tags)
  };
};
const savedObjectClientFromRequest = async ctx => {
  if (!ctx.requestHandlerContext) {
    throw new Error('Storage context.requestHandlerContext missing.');
  }
  const {
    savedObjects
  } = await ctx.requestHandlerContext.core;
  return savedObjects.client;
};
function savedObjectToLensSavedObject(savedObject) {
  const {
    id,
    type,
    updated_at: updatedAt,
    created_at: createdAt,
    attributes: {
      title,
      description,
      state,
      visualizationType
    },
    references,
    error,
    namespaces
  } = savedObject;
  return {
    id,
    type,
    updatedAt,
    createdAt,
    attributes: {
      title,
      description,
      visualizationType,
      state
    },
    references,
    error,
    namespaces
  };
}
class LensStorage extends _contentManagementUtils.SOContentStorage {
  constructor(params) {
    super({
      savedObjectType: _content_management.CONTENT_ID,
      cmServicesDefinition: _cm_services.cmServicesDefinition,
      searchArgsToSOFindOptions,
      enableMSearch: true,
      allowedSavedObjectAttributes: ['title', 'description', 'visualizationType', 'state'],
      logger: params.logger,
      throwOnResultValidationError: params.throwOnResultValidationError
    });
    this.params = params;
  }

  /**
   * Lens requires a custom update function because of https://github.com/elastic/kibana/issues/160116
   * where a forced create with overwrite flag is used instead of regular update
   */
  async update(ctx, id, data, options) {
    const {
      utils: {
        getTransforms
      },
      version: {
        request: requestVersion
      }
    } = ctx;
    const transforms = getTransforms(_cm_services.cmServicesDefinition, requestVersion);

    // Validate input (data & options) & UP transform them to the latest version
    const {
      value: dataToLatest,
      error: dataError
    } = transforms.update.in.data.up(data);
    if (dataError) {
      throw _boom.default.badRequest(`Invalid data. ${dataError.message}`);
    }
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.update.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid options. ${optionsError.message}`);
    }

    // Save data in DB
    const soClient = await savedObjectClientFromRequest(ctx);
    const savedObject = await soClient.create(_content_management.CONTENT_ID, dataToLatest, {
      id,
      overwrite: true,
      ...optionsToLatest
    });
    const result = {
      item: savedObjectToLensSavedObject(savedObject)
    };
    const validationError = transforms.update.out.result.validate(result);
    if (validationError) {
      if (this.params.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.params.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate DB response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.update.out.result.down(result, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );

    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
}
exports.LensStorage = LensStorage;