"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.annotationLayerHasUnsavedChanges = void 0;
exports.getAnnotationLayerErrors = getAnnotationLayerErrors;
exports.getColumnToLabelMap = void 0;
exports.getIconForSeries = getIconForSeries;
exports.getPersistableState = getPersistableState;
exports.getSeriesColor = void 0;
exports.hasHistogramSeries = hasHistogramSeries;
exports.injectReferences = injectReferences;
exports.isHorizontalChart = isHorizontalChart;
exports.isHorizontalSeries = isHorizontalSeries;
exports.isPercentageSeries = isPercentageSeries;
exports.isPersistedState = isPersistedState;
exports.isStackedChart = isStackedChart;
var _eventAnnotationCommon = require("@kbn/event-annotation-common");
var _uuid = require("uuid");
var _eventAnnotationComponents = require("@kbn/event-annotation-components");
var _i18n = require("@kbn/i18n");
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _lodash = require("lodash");
var _visualizationUiComponents = require("@kbn/visualization-ui-components");
var _types = require("./types");
var _visualization_helpers = require("./visualization_helpers");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isHorizontalSeries(seriesType) {
  return seriesType === 'bar_horizontal' || seriesType === 'bar_horizontal_stacked' || seriesType === 'bar_horizontal_percentage_stacked';
}
function isPercentageSeries(seriesType) {
  return seriesType === 'bar_percentage_stacked' || seriesType === 'bar_horizontal_percentage_stacked' || seriesType === 'area_percentage_stacked';
}
function isStackedChart(seriesType) {
  return seriesType.includes('stacked');
}
function isHorizontalChart(layers) {
  return (0, _visualization_helpers.getDataLayers)(layers).every(l => isHorizontalSeries(l.seriesType));
}
function getIconForSeries(type) {
  const definition = _types.visualizationTypes.find(t => t.id === type);
  if (!definition) {
    throw new Error(`Unknown series type ${type}`);
  }
  return definition.icon || 'empty';
}
const getSeriesColor = (layer, accessor) => {
  var _layer$yConfig, _layer$yConfig$find;
  if ((0, _visualization_helpers.isAnnotationsLayer)(layer)) {
    var _layer$annotations, _layer$annotations$fi;
    return (layer === null || layer === void 0 ? void 0 : (_layer$annotations = layer.annotations) === null || _layer$annotations === void 0 ? void 0 : (_layer$annotations$fi = _layer$annotations.find(ann => ann.id === accessor)) === null || _layer$annotations$fi === void 0 ? void 0 : _layer$annotations$fi.color) || null;
  }
  if ((0, _visualization_helpers.isDataLayer)(layer) && layer.splitAccessor && !layer.collapseFn) {
    return null;
  }
  return (layer === null || layer === void 0 ? void 0 : (_layer$yConfig = layer.yConfig) === null || _layer$yConfig === void 0 ? void 0 : (_layer$yConfig$find = _layer$yConfig.find(yConfig => yConfig.forAccessor === accessor)) === null || _layer$yConfig$find === void 0 ? void 0 : _layer$yConfig$find.color) || null;
};
exports.getSeriesColor = getSeriesColor;
const getColumnToLabelMap = (layer, datasource) => {
  const columnToLabel = {};
  layer.accessors.concat((0, _visualization_helpers.isDataLayer)(layer) && layer.splitAccessor ? [layer.splitAccessor] : []).forEach(accessor => {
    const operation = datasource === null || datasource === void 0 ? void 0 : datasource.getOperationForColumnId(accessor);
    if (operation !== null && operation !== void 0 && operation.label) {
      columnToLabel[accessor] = operation.label;
    }
  });
  return columnToLabel;
};
exports.getColumnToLabelMap = getColumnToLabelMap;
function hasHistogramSeries(layers = [], datasourceLayers) {
  if (!datasourceLayers) {
    return false;
  }
  const validLayers = layers.filter(({
    accessors
  }) => accessors.length);
  return validLayers.some(({
    layerId,
    xAccessor
  }) => {
    var _datasourceLayers$lay;
    if (!xAccessor) {
      return false;
    }
    const xAxisOperation = (_datasourceLayers$lay = datasourceLayers[layerId]) === null || _datasourceLayers$lay === void 0 ? void 0 : _datasourceLayers$lay.getOperationForColumnId(xAccessor);
    return xAxisOperation && xAxisOperation.isBucketed && xAxisOperation.scale && xAxisOperation.scale !== 'ordinal';
  });
}
function getLayerReferenceName(layerId) {
  return `xy-visualization-layer-${layerId}`;
}
const annotationLayerHasUnsavedChanges = layer => {
  if (!(0, _visualization_helpers.isByReferenceAnnotationsLayer)(layer)) {
    return false;
  }
  const currentConfig = {
    annotations: layer.annotations,
    ignoreGlobalFilters: layer.ignoreGlobalFilters,
    indexPatternId: layer.indexPatternId
  };
  const savedConfig = {
    annotations: layer.__lastSaved.annotations,
    ignoreGlobalFilters: layer.__lastSaved.ignoreGlobalFilters,
    indexPatternId: layer.__lastSaved.indexPatternId
  };
  return !(0, _fastDeepEqual.default)(currentConfig, savedConfig);
};
exports.annotationLayerHasUnsavedChanges = annotationLayerHasUnsavedChanges;
function getPersistableState(state) {
  const savedObjectReferences = [];
  const persistableLayers = [];
  state.layers.forEach(layer => {
    if (!(0, _visualization_helpers.isAnnotationsLayer)(layer)) {
      persistableLayers.push(layer);
    } else {
      if ((0, _visualization_helpers.isByReferenceAnnotationsLayer)(layer)) {
        const referenceName = `ref-${(0, _uuid.v4)()}`;
        savedObjectReferences.push({
          type: _eventAnnotationCommon.EVENT_ANNOTATION_GROUP_TYPE,
          id: layer.annotationGroupId,
          name: referenceName
        });
        if (!annotationLayerHasUnsavedChanges(layer)) {
          const persistableLayer = {
            persistanceType: 'byReference',
            layerId: layer.layerId,
            layerType: layer.layerType,
            annotationGroupRef: referenceName
          };
          persistableLayers.push(persistableLayer);
        } else {
          const persistableLayer = {
            persistanceType: 'linked',
            cachedMetadata: layer.cachedMetadata || {
              title: layer.__lastSaved.title,
              description: layer.__lastSaved.description,
              tags: layer.__lastSaved.tags
            },
            layerId: layer.layerId,
            layerType: layer.layerType,
            annotationGroupRef: referenceName,
            annotations: layer.annotations,
            ignoreGlobalFilters: layer.ignoreGlobalFilters
          };
          persistableLayers.push(persistableLayer);
          savedObjectReferences.push({
            type: 'index-pattern',
            id: layer.indexPatternId,
            name: getLayerReferenceName(layer.layerId)
          });
        }
      } else {
        const {
          indexPatternId,
          ...persistableLayer
        } = layer;
        savedObjectReferences.push({
          type: 'index-pattern',
          id: indexPatternId,
          name: getLayerReferenceName(layer.layerId)
        });
        persistableLayers.push({
          ...persistableLayer,
          persistanceType: 'byValue'
        });
      }
    }
  });
  return {
    savedObjectReferences,
    state: {
      ...state,
      layers: persistableLayers
    }
  };
}
function isPersistedState(state) {
  return state.layers.some(_visualization_helpers.isPersistedAnnotationsLayer);
}
function injectReferences(state, annotationGroups, references) {
  if (!references || !references.length) {
    return state;
  }
  if (!annotationGroups) {
    throw new Error('xy visualization: injecting references relies on annotation groups but they were not provided');
  }

  // called on-demand since indexPattern reference won't be here on the vis if its a by-reference group
  const getIndexPatternIdFromReferences = annotationLayerId => {
    var _references$find;
    const fallbackIndexPatternId = references.find(({
      type
    }) => type === 'index-pattern').id;
    return ((_references$find = references.find(({
      name
    }) => name === getLayerReferenceName(annotationLayerId))) === null || _references$find === void 0 ? void 0 : _references$find.id) || fallbackIndexPatternId;
  };
  return {
    ...state,
    layers: state.layers.map(persistedLayer => {
      if (!(0, _visualization_helpers.isPersistedAnnotationsLayer)(persistedLayer)) {
        return persistedLayer;
      }
      let injectedLayer;
      if ((0, _visualization_helpers.isPersistedByValueAnnotationsLayer)(persistedLayer)) {
        injectedLayer = {
          ...persistedLayer,
          indexPatternId: getIndexPatternIdFromReferences(persistedLayer.layerId)
        };
      } else {
        var _references$find2;
        const annotationGroupId = references === null || references === void 0 ? void 0 : (_references$find2 = references.find(({
          name
        }) => name === persistedLayer.annotationGroupRef)) === null || _references$find2 === void 0 ? void 0 : _references$find2.id;
        const annotationGroup = annotationGroupId ? annotationGroups[annotationGroupId] : undefined;
        if (!annotationGroupId || !annotationGroup) {
          return undefined; // the annotation group this layer was referencing is gone, so remove the layer
        }

        // declared as a separate variable for type checking
        const commonProps = {
          layerId: persistedLayer.layerId,
          layerType: persistedLayer.layerType,
          annotationGroupId,
          __lastSaved: annotationGroup
        };
        if ((0, _visualization_helpers.isPersistedByReferenceAnnotationsLayer)(persistedLayer)) {
          // a clean by-reference layer inherits everything from the library annotation group
          injectedLayer = {
            ...commonProps,
            ignoreGlobalFilters: annotationGroup.ignoreGlobalFilters,
            indexPatternId: annotationGroup.indexPatternId,
            annotations: (0, _lodash.cloneDeep)(annotationGroup.annotations)
          };
        } else {
          // a linked by-value layer gets settings from visualization state while
          // still maintaining the reference to the library annotation group
          injectedLayer = {
            ...commonProps,
            ignoreGlobalFilters: persistedLayer.ignoreGlobalFilters,
            indexPatternId: getIndexPatternIdFromReferences(persistedLayer.layerId),
            annotations: (0, _lodash.cloneDeep)(persistedLayer.annotations),
            cachedMetadata: persistedLayer.cachedMetadata
          };
        }
      }
      return injectedLayer;
    }).filter(_utils.nonNullable)
  };
}
function getAnnotationLayerErrors(layer, columnId, dataViews) {
  var _annotation$extraFiel;
  if (!layer) {
    return [];
  }
  const annotation = layer.annotations.find(({
    id
  }) => id === columnId);
  if (!annotation || !(0, _eventAnnotationComponents.isQueryAnnotationConfig)(annotation)) {
    return [];
  }
  const layerDataView = dataViews.indexPatterns[layer.indexPatternId];
  const invalidMessages = [];
  if (annotation.timeField == null || annotation.timeField === '') {
    invalidMessages.push(_i18n.i18n.translate('xpack.lens.xyChart.annotationError.timeFieldEmpty', {
      defaultMessage: 'Time field is missing'
    }));
  }
  if (annotation.timeField && !Boolean(layerDataView.getFieldByName(annotation.timeField))) {
    invalidMessages.push(_i18n.i18n.translate('xpack.lens.xyChart.annotationError.timeFieldNotFound', {
      defaultMessage: 'Time field {timeField} not found in data view {dataView}',
      values: {
        timeField: annotation.timeField,
        dataView: layerDataView.title
      }
    }));
  }
  const {
    isValid,
    error
  } = (0, _visualizationUiComponents.validateQuery)(annotation === null || annotation === void 0 ? void 0 : annotation.filter, layerDataView);
  if (!isValid && error) {
    invalidMessages.push(error);
  }
  if (annotation.textField && !Boolean(layerDataView.getFieldByName(annotation.textField))) {
    invalidMessages.push(_i18n.i18n.translate('xpack.lens.xyChart.annotationError.textFieldNotFound', {
      defaultMessage: 'Text field {textField} not found in data view {dataView}',
      values: {
        textField: annotation.textField,
        dataView: layerDataView.title
      }
    }));
  }
  if ((_annotation$extraFiel = annotation.extraFields) !== null && _annotation$extraFiel !== void 0 && _annotation$extraFiel.length) {
    const missingTooltipFields = [];
    for (const field of annotation.extraFields) {
      if (!Boolean(layerDataView.getFieldByName(field))) {
        missingTooltipFields.push(field);
      }
    }
    if (missingTooltipFields.length) {
      invalidMessages.push(_i18n.i18n.translate('xpack.lens.xyChart.annotationError.tooltipFieldNotFound', {
        defaultMessage: 'Tooltip {missingFields, plural, one {field} other {fields}} {missingTooltipFields} not found in data view {dataView}',
        values: {
          missingTooltipFields: missingTooltipFields.join(', '),
          missingFields: missingTooltipFields.length,
          dataView: layerDataView.title
        }
      }));
    }
  }
  return invalidMessages;
}