"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataBounds = getDataBounds;
exports.hasNumericHistogramDimension = hasNumericHistogramDimension;
exports.validateAxisDomain = validateAxisDomain;
exports.validateExtent = validateExtent;
exports.validateZeroInclusivityExtent = validateZeroInclusivityExtent;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns true if the provided extent includes 0
 * @param extent
 * @returns boolean
 */
function validateZeroInclusivityExtent(extent) {
  return extent && extent.lowerBound != null && extent.upperBound != null && extent.lowerBound <= 0 && extent.upperBound >= 0;
}

/**
 * Returns true if the provided extent is a valid range
 * @param extent
 * @returns boolean
 */
function validateAxisDomain(extents) {
  return extents && extents.lowerBound != null && extents.upperBound != null && extents.upperBound > extents.lowerBound;
}
/**
 * Returns true if the provided column is a numeric histogram dimension
 * @param extent
 * @returns boolean
 */
function hasNumericHistogramDimension(datasourceLayer, columnId) {
  if (!columnId) {
    return false;
  }
  const operation = datasourceLayer === null || datasourceLayer === void 0 ? void 0 : datasourceLayer.getOperationForColumnId(columnId);
  return Boolean(operation && operation.dataType === 'number' && operation.scale === 'interval');
}

/**
 * Finds the table data min and max. Returns undefined when no min/max is found
 * @param layerId
 * @param tables
 * @param columnId
 * @returns
 */
function getDataBounds(layerId, tables, columnId) {
  const table = tables === null || tables === void 0 ? void 0 : tables[layerId];
  if (columnId && table) {
    const sortedRows = table.rows.map(({
      [columnId]: value
    }) => value)
    // ignore missing hit
    .filter(v => v != null).sort((a, b) => a - b);
    if (sortedRows.length) {
      return {
        min: sortedRows[0],
        max: sortedRows[sortedRows.length - 1]
      };
    }
  }
}
function validateExtent(shouldIncludeZero, extent) {
  return {
    inclusiveZeroError: shouldIncludeZero && !validateZeroInclusivityExtent(extent),
    boundaryError: !validateAxisDomain(extent)
  };
}