"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildDatasourceStates = exports.addLayerFormulaColumns = exports.addLayerColumn = void 0;
exports.buildReferences = buildReferences;
exports.getAdhocDataviews = void 0;
exports.getDataView = getDataView;
exports.getDatasetIndex = getDatasetIndex;
exports.getDefaultReferences = void 0;
exports.isFormulaDataset = isFormulaDataset;
var _uuid = require("uuid");
var _esQuery = require("@kbn/es-query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const getDefaultReferences = (index, dataLayerId) => {
  return [{
    type: 'index-pattern',
    id: index,
    name: `indexpattern-datasource-layer-${dataLayerId}`
  }];
};
exports.getDefaultReferences = getDefaultReferences;
function buildReferences(dataviews) {
  const references = [];
  for (const layerid in dataviews) {
    if (dataviews[layerid]) {
      references.push(...getDefaultReferences(dataviews[layerid].id, layerid));
    }
  }
  return references.flat();
}
const getAdhocDataView = dataView => {
  var _dataView$id;
  return {
    [(_dataView$id = dataView.id) !== null && _dataView$id !== void 0 ? _dataView$id : (0, _uuid.v4)()]: {
      ...dataView.toSpec()
    }
  };
};
const getAdhocDataviews = dataviews => {
  let adHocDataViews = {};
  [...new Set(Object.values(dataviews))].forEach(d => {
    adHocDataViews = {
      ...adHocDataViews,
      ...getAdhocDataView(d)
    };
  });
  return adHocDataViews;
};
exports.getAdhocDataviews = getAdhocDataviews;
function isFormulaDataset(dataset) {
  if (dataset && 'index' in dataset) {
    return true;
  }
  return false;
}

/**
 * it loads dataview by id or creates an ad-hoc dataview if index pattern is provided
 * @param index
 * @param dataViewsAPI
 * @param timeField
 */
async function getDataView(index, dataViewsAPI, timeField) {
  let dataView;
  try {
    dataView = await dataViewsAPI.get(index, false);
  } catch {
    dataView = await dataViewsAPI.create({
      title: index,
      timeFieldName: timeField || '@timestamp'
    });
  }
  return dataView;
}
function getDatasetIndex(dataset) {
  if (!dataset) return undefined;
  let index;
  let timeFieldName = '@timestamp';
  if ('index' in dataset) {
    index = dataset.index;
    timeFieldName = dataset.timeFieldName || '@timestamp';
  } else if ('esql' in dataset) {
    index = (0, _esQuery.getIndexPatternFromESQLQuery)(dataset.esql); // parseIndexFromQuery(config.dataset.query);
  } else {
    return undefined;
  }
  return {
    index,
    timeFieldName
  };
}
function buildDatasourceStatesLayer(layer, i, dataset, dataView, buildFormulaLayers, getValueColumns) {
  function buildValueLayer(config) {
    const table = dataset;
    const newLayer = {
      table,
      columns: getValueColumns(layer, i),
      allColumns: table.columns.map(column => ({
        fieldName: column.name,
        columnId: column.id,
        meta: column.meta
      })),
      index: '',
      query: undefined
    };
    return newLayer;
  }
  function buildESQLLayer(config) {
    const columns = getValueColumns(layer, i);
    const newLayer = {
      index: dataView.id,
      query: {
        esql: dataset.esql
      },
      columns,
      allColumns: columns
    };
    return newLayer;
  }
  if ('esql' in dataset) {
    return ['textBased', buildESQLLayer(layer)];
  } else if ('type' in dataset) {
    return ['textBased', buildValueLayer(layer)];
  }
  return ['formBased', buildFormulaLayers(layer, i, dataView)];
}
const buildDatasourceStates = async (config, dataviews, buildFormulaLayers, getValueColumns, dataViewsAPI) => {
  const layers = {
    textBased: {
      layers: {}
    },
    formBased: {
      layers: {}
    }
  };
  const mainDataset = config.dataset;
  const configLayers = 'layers' in config ? config.layers : [config];
  for (let i = 0; i < configLayers.length; i++) {
    const layer = configLayers[i];
    const layerId = `layer_${i}`;
    const dataset = layer.dataset || mainDataset;
    if (!dataset && 'type' in layer && layer.type !== 'annotation') {
      throw Error('dataset must be defined');
    }
    const index = getDatasetIndex(dataset);
    const dataView = index ? await getDataView(index.index, dataViewsAPI, index.timeFieldName) : undefined;
    if (dataView) {
      dataviews[layerId] = dataView;
    }
    if (dataset) {
      const [type, layerConfig] = buildDatasourceStatesLayer(layer, i, dataset, dataView, buildFormulaLayers, getValueColumns);
      if (layerConfig) {
        layers[type].layers[layerId] = layerConfig;
      }
    }
  }
  return layers;
};
exports.buildDatasourceStates = buildDatasourceStates;
const addLayerColumn = (layer, columnName, config, first = false) => {
  layer.columns = {
    ...layer.columns,
    [columnName]: config
  };
  if (first) {
    layer.columnOrder.unshift(columnName);
  } else {
    layer.columnOrder.push(columnName);
  }
};
exports.addLayerColumn = addLayerColumn;
const addLayerFormulaColumns = (layer, columns, postfix = '') => {
  const altObj = Object.fromEntries(Object.entries(columns.columns).map(([key, value]) =>
  // Modify key here
  [`${key}${postfix}`, value]));
  layer.columns = {
    ...layer.columns,
    ...altObj
  };
  layer.columnOrder.push(...columns.columnOrder.map(c => `${c}${postfix}`));
};
exports.addLayerFormulaColumns = addLayerFormulaColumns;