"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMetricsHostsAnomaliesResults = exports.callGetMetricHostsAnomaliesAPI = void 0;
var _react = require("react");
var _infra_ml = require("../../../../../common/http_api/infra_ml");
var _use_tracked_promise = require("../../../../utils/use_tracked_promise");
var _runtime_types = require("../../../../../common/runtime_types");
var _use_kibana = require("../../../../hooks/use_kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const stateReducer = (state, action) => {
  const resetPagination = {
    page: 1,
    paginationCursor: undefined
  };
  switch (action.type) {
    case 'changePaginationOptions':
      return {
        ...state,
        ...resetPagination,
        ...action.payload
      };
    case 'changeSortOptions':
      return {
        ...state,
        ...resetPagination,
        ...action.payload
      };
    case 'changeHasNextPage':
      return {
        ...state,
        ...action.payload
      };
    case 'changeLastReceivedCursors':
      return {
        ...state,
        ...action.payload
      };
    case 'fetchNextPage':
      return state.lastReceivedCursors ? {
        ...state,
        page: state.page + 1,
        paginationCursor: {
          searchAfter: state.lastReceivedCursors.nextPageCursor
        }
      } : state;
    case 'fetchPreviousPage':
      return state.lastReceivedCursors ? {
        ...state,
        page: state.page - 1,
        paginationCursor: {
          searchBefore: state.lastReceivedCursors.previousPageCursor
        }
      } : state;
    case 'changeTimeRange':
      return {
        ...state,
        ...resetPagination,
        ...action.payload
      };
    case 'changeFilteredDatasets':
      return {
        ...state,
        ...resetPagination,
        ...action.payload
      };
    default:
      return state;
  }
};
const STATE_DEFAULTS = {
  // NOTE: This piece of state is purely for the client side, it could be extracted out of the hook.
  page: 1,
  // Cursor from the last request
  lastReceivedCursors: undefined,
  // Cursor to use for the next request. For the first request, and therefore not paging, this will be undefined.
  paginationCursor: undefined,
  hasNextPage: false
};
const initStateReducer = (endTime, startTime, defaultSortOptions, defaultPaginationOptions, filteredDatasets) => stateDefaults => {
  return {
    ...stateDefaults,
    paginationOptions: defaultPaginationOptions,
    sortOptions: defaultSortOptions,
    filteredDatasets,
    timeRange: {
      start: startTime,
      end: endTime
    }
  };
};
const useMetricsHostsAnomaliesResults = ({
  endTime,
  startTime,
  sourceId,
  anomalyThreshold,
  defaultSortOptions,
  defaultPaginationOptions,
  onGetMetricsHostsAnomaliesDatasetsError,
  filteredDatasets
}) => {
  const {
    services
  } = (0, _use_kibana.useKibanaContextForPlugin)();
  const abortController = (0, _react.useRef)(new AbortController());
  const [reducerState, dispatch] = (0, _react.useReducer)(stateReducer, STATE_DEFAULTS, initStateReducer(endTime, startTime, defaultSortOptions, defaultPaginationOptions, filteredDatasets));
  const [metricsHostsAnomalies, setMetricsHostsAnomalies] = (0, _react.useState)([]);
  (0, _react.useEffect)(() => {
    const current = abortController === null || abortController === void 0 ? void 0 : abortController.current;
    return () => {
      current.abort();
    };
  }, []);
  const [getMetricsHostsAnomaliesRequest, getMetricsHostsAnomalies] = (0, _use_tracked_promise.useTrackedPromise)({
    cancelPreviousOn: 'creation',
    createPromise: async (metric, query, hostName) => {
      const {
        timeRange: {
          start: queryStartTime,
          end: queryEndTime
        },
        sortOptions,
        paginationOptions,
        paginationCursor
      } = reducerState;
      abortController.current.abort();
      abortController.current = new AbortController();
      return await callGetMetricHostsAnomaliesAPI({
        sourceId,
        anomalyThreshold,
        startTime: queryStartTime,
        endTime: queryEndTime,
        metric,
        query,
        sort: sortOptions,
        pagination: {
          ...paginationOptions,
          cursor: paginationCursor
        },
        hostName
      }, services.http.fetch, abortController.current.signal);
    },
    onResolve: ({
      data: {
        anomalies,
        paginationCursors: requestCursors,
        hasMoreEntries
      }
    }) => {
      const {
        paginationCursor
      } = reducerState;
      if (requestCursors) {
        dispatch({
          type: 'changeLastReceivedCursors',
          payload: {
            lastReceivedCursors: requestCursors
          }
        });
      }

      // Check if we have more "next" entries. "Page" covers the "previous" scenario,
      // since we need to know the page we're on anyway.
      if (!paginationCursor || paginationCursor && 'searchAfter' in paginationCursor) {
        dispatch({
          type: 'changeHasNextPage',
          payload: {
            hasNextPage: hasMoreEntries
          }
        });
      } else if (paginationCursor && 'searchBefore' in paginationCursor) {
        // We've requested a previous page, therefore there is a next page.
        dispatch({
          type: 'changeHasNextPage',
          payload: {
            hasNextPage: true
          }
        });
      }
      setMetricsHostsAnomalies(anomalies);
    }
  }, [sourceId, anomalyThreshold, dispatch, reducerState.timeRange.start, reducerState.timeRange.end, reducerState.sortOptions, reducerState.paginationOptions, reducerState.paginationCursor, reducerState.filteredDatasets]);
  const changeSortOptions = (0, _react.useCallback)(nextSortOptions => {
    dispatch({
      type: 'changeSortOptions',
      payload: {
        sortOptions: nextSortOptions
      }
    });
  }, [dispatch]);
  const changePaginationOptions = (0, _react.useCallback)(nextPaginationOptions => {
    dispatch({
      type: 'changePaginationOptions',
      payload: {
        paginationOptions: nextPaginationOptions
      }
    });
  }, [dispatch]);

  // Time range has changed
  (0, _react.useEffect)(() => {
    dispatch({
      type: 'changeTimeRange',
      payload: {
        timeRange: {
          start: startTime,
          end: endTime
        }
      }
    });
  }, [startTime, endTime]);

  // Selected datasets have changed
  (0, _react.useEffect)(() => {
    dispatch({
      type: 'changeFilteredDatasets',
      payload: {
        filteredDatasets
      }
    });
  }, [filteredDatasets]);
  const handleFetchNextPage = (0, _react.useCallback)(() => {
    if (reducerState.lastReceivedCursors) {
      dispatch({
        type: 'fetchNextPage'
      });
    }
  }, [dispatch, reducerState]);
  const handleFetchPreviousPage = (0, _react.useCallback)(() => {
    if (reducerState.lastReceivedCursors) {
      dispatch({
        type: 'fetchPreviousPage'
      });
    }
  }, [dispatch, reducerState]);
  const isLoadingMetricsHostsAnomalies = (0, _react.useMemo)(() => getMetricsHostsAnomaliesRequest.state === 'pending', [getMetricsHostsAnomaliesRequest.state]);
  const hasFailedLoadingMetricsHostsAnomalies = (0, _react.useMemo)(() => getMetricsHostsAnomaliesRequest.state === 'rejected', [getMetricsHostsAnomaliesRequest.state]);
  return {
    metricsHostsAnomalies,
    getMetricsHostsAnomalies,
    isLoadingMetricsHostsAnomalies,
    hasFailedLoadingMetricsHostsAnomalies,
    changeSortOptions,
    sortOptions: reducerState.sortOptions,
    changePaginationOptions,
    paginationOptions: reducerState.paginationOptions,
    fetchPreviousPage: reducerState.page > 1 ? handleFetchPreviousPage : undefined,
    fetchNextPage: reducerState.hasNextPage ? handleFetchNextPage : undefined,
    page: reducerState.page,
    timeRange: reducerState.timeRange
  };
};
exports.useMetricsHostsAnomaliesResults = useMetricsHostsAnomaliesResults;
const callGetMetricHostsAnomaliesAPI = async (requestArgs, fetch, signal) => {
  const {
    sourceId,
    anomalyThreshold,
    startTime,
    endTime,
    metric,
    sort,
    pagination,
    query,
    hostName
  } = requestArgs;
  const response = await fetch(_infra_ml.INFA_ML_GET_METRICS_HOSTS_ANOMALIES_PATH, {
    method: 'POST',
    body: JSON.stringify(_infra_ml.getMetricsHostsAnomaliesRequestPayloadRT.encode({
      data: {
        sourceId,
        anomalyThreshold,
        timeRange: {
          startTime,
          endTime
        },
        query,
        metric,
        sort,
        pagination,
        hostName
      }
    })),
    signal
  });
  return (0, _runtime_types.decodeOrThrow)(_infra_ml.getMetricsHostsAnomaliesSuccessReponsePayloadRT)(response);
};
exports.callGetMetricHostsAnomaliesAPI = callGetMetricHostsAnomaliesAPI;