"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUpgradingSteps = exports.getAgentsPerVersion = exports.fetchAgentMetrics = void 0;
var _common = require("../../../common");
var _agent_collectors = require("../../collectors/agent_collectors");
var _helpers = require("../../collectors/helpers");
var _app_context = require("../app_context");
var _retry = require("../epm/elasticsearch/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fetchAgentMetrics = async (core, abortController) => {
  const [soClient, esClient] = await (0, _helpers.getInternalClients)(core);
  if (!soClient || !esClient) {
    return;
  }
  const fleetAgentsIndexExists = await esClient.indices.get({
    index: _common.AGENTS_INDEX
  }).catch(error => {
    if (error.statusCode === 404) {
      return;
    }
    throw error;
  });
  if (!fleetAgentsIndexExists) {
    return;
  }
  const usage = {
    agents: await (0, _agent_collectors.getAgentUsage)(soClient, esClient),
    agents_per_version: await getAgentsPerVersion(esClient, abortController),
    upgrading_step: await getUpgradingSteps(esClient, abortController)
  };
  return usage;
};
exports.fetchAgentMetrics = fetchAgentMetrics;
const getAgentsPerVersion = async (esClient, abortController) => {
  try {
    var _buckets, _response$aggregation;
    const response = await (0, _retry.retryTransientEsErrors)(() => esClient.search({
      index: _common.AGENTS_INDEX,
      query: {
        bool: {
          filter: [{
            term: {
              active: 'true'
            }
          }]
        }
      },
      size: 0,
      aggs: {
        versions: {
          terms: {
            field: 'agent.version'
          }
        }
      }
    }, {
      signal: abortController.signal
    }));
    return ((_buckets = (response === null || response === void 0 ? void 0 : (_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : _response$aggregation.versions).buckets) !== null && _buckets !== void 0 ? _buckets : []).map(bucket => ({
      version: bucket.key,
      count: bucket.doc_count
    }));
  } catch (error) {
    if (error.statusCode === 404) {
      _app_context.appContextService.getLogger().debug('Index .fleet-agents does not exist yet.');
    } else {
      throw error;
    }
    return [];
  }
};
exports.getAgentsPerVersion = getAgentsPerVersion;
const getUpgradingSteps = async (esClient, abortController) => {
  const upgradingSteps = {
    requested: 0,
    scheduled: 0,
    downloading: 0,
    extracting: 0,
    replacing: 0,
    restarting: 0,
    watching: 0,
    rollback: 0,
    failed: 0
  };
  try {
    var _buckets2, _response$aggregation2;
    const response = await (0, _retry.retryTransientEsErrors)(() => esClient.search({
      index: _common.AGENTS_INDEX,
      query: {
        bool: {
          filter: [{
            term: {
              active: 'true'
            }
          }]
        }
      },
      size: 0,
      aggs: {
        upgrade_details: {
          terms: {
            field: 'upgrade_details.state'
          }
        }
      }
    }, {
      signal: abortController.signal
    }));
    ((_buckets2 = (response === null || response === void 0 ? void 0 : (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.upgrade_details).buckets) !== null && _buckets2 !== void 0 ? _buckets2 : []).forEach(bucket => {
      switch (bucket.key) {
        case 'UPG_REQUESTED':
          upgradingSteps.requested = bucket.doc_count;
          break;
        case 'UPG_SCHEDULED':
          upgradingSteps.scheduled = bucket.doc_count;
          break;
        case 'UPG_DOWNLOADING':
          upgradingSteps.downloading = bucket.doc_count;
          break;
        case 'UPG_EXTRACTING':
          upgradingSteps.extracting = bucket.doc_count;
          break;
        case 'UPG_REPLACING':
          upgradingSteps.replacing = bucket.doc_count;
          break;
        case 'UPG_RESTARTING':
          upgradingSteps.restarting = bucket.doc_count;
          break;
        case 'UPG_WATCHING':
          upgradingSteps.watching = bucket.doc_count;
          break;
        case 'UPG_ROLLBACK':
          upgradingSteps.rollback = bucket.doc_count;
          break;
        case 'UPG_FAILED':
          upgradingSteps.failed = bucket.doc_count;
          break;
        default:
          break;
      }
    });
    return upgradingSteps;
  } catch (error) {
    if (error.statusCode === 404) {
      _app_context.appContextService.getLogger().debug('Index .fleet-agents does not exist yet.');
    } else {
      throw error;
    }
    return upgradingSteps;
  }
};
exports.getUpgradingSteps = getUpgradingSteps;