"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFileHandler = void 0;
var _path = _interopRequireDefault(require("path"));
var _mimeTypes = _interopRequireDefault(require("mime-types"));
var _packages = require("../../services/epm/packages");
var _errors = require("../../errors");
var _archive = require("../../services/epm/archive");
var _storage = require("../../services/epm/archive/storage");
var _bundled_packages = require("../../services/epm/packages/bundled_packages");
var _registry = require("../../services/epm/registry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CACHE_CONTROL_10_MINUTES_HEADER = {
  'cache-control': 'max-age=600'
};
const getFileHandler = async (context, request, response) => {
  try {
    const {
      pkgName,
      pkgVersion,
      filePath
    } = request.params;
    const savedObjectsClient = (await context.fleet).internalSoClient;
    const installation = await (0, _packages.getInstallation)({
      savedObjectsClient,
      pkgName
    });
    const useLocalFile = pkgVersion === (installation === null || installation === void 0 ? void 0 : installation.version);
    const assetPath = `${pkgName}-${pkgVersion}/${filePath}`;
    if (useLocalFile) {
      const fileBuffer = (0, _archive.getArchiveEntry)(assetPath);
      // only pull local installation if we don't have it cached
      const storedAsset = !fileBuffer && (await (0, _storage.getAsset)({
        savedObjectsClient,
        path: assetPath
      }));

      // error, if neither is available
      if (!fileBuffer && !storedAsset) {
        return response.custom({
          body: `installed package file not found: ${filePath}`,
          statusCode: 404
        });
      }

      // if storedAsset is not available, fileBuffer *must* be
      // b/c we error if we don't have at least one, and storedAsset is the least likely
      const {
        buffer,
        contentType
      } = storedAsset ? {
        contentType: storedAsset.media_type,
        buffer: storedAsset.data_utf8 ? Buffer.from(storedAsset.data_utf8, 'utf8') : Buffer.from(storedAsset.data_base64, 'base64')
      } : {
        contentType: _mimeTypes.default.contentType(_path.default.extname(assetPath)),
        buffer: fileBuffer
      };
      if (!contentType) {
        return response.custom({
          body: `unknown content type for file: ${filePath}`,
          statusCode: 400
        });
      }
      return response.custom({
        body: buffer,
        statusCode: 200,
        headers: {
          ...CACHE_CONTROL_10_MINUTES_HEADER,
          'content-type': contentType
        }
      });
    }
    const bundledPackage = await (0, _bundled_packages.getBundledPackageByPkgKey)((0, _registry.pkgToPkgKey)({
      name: pkgName,
      version: pkgVersion
    }));
    if (bundledPackage) {
      var _bufferEntries$find;
      const bufferEntries = await (0, _archive.unpackBufferEntries)(await bundledPackage.getBuffer(), 'application/zip');
      const fileBuffer = (_bufferEntries$find = bufferEntries.find(entry => entry.path === assetPath)) === null || _bufferEntries$find === void 0 ? void 0 : _bufferEntries$find.buffer;
      if (!fileBuffer) {
        return response.custom({
          body: `bundled package file not found: ${filePath}`,
          statusCode: 404
        });
      }

      // if storedAsset is not available, fileBuffer *must* be
      // b/c we error if we don't have at least one, and storedAsset is the least likely
      const {
        buffer,
        contentType
      } = {
        contentType: _mimeTypes.default.contentType(_path.default.extname(assetPath)),
        buffer: fileBuffer
      };
      if (!contentType) {
        return response.custom({
          body: `unknown content type for file: ${filePath}`,
          statusCode: 400
        });
      }
      return response.custom({
        body: buffer,
        statusCode: 200,
        headers: {
          ...CACHE_CONTROL_10_MINUTES_HEADER,
          'content-type': contentType
        }
      });
    } else {
      const registryResponse = await (0, _packages.getFile)(pkgName, pkgVersion, filePath);
      const headersToProxy = ['content-type'];
      const proxiedHeaders = headersToProxy.reduce((headers, knownHeader) => {
        const value = registryResponse.headers.get(knownHeader);
        if (value !== null) {
          headers[knownHeader] = value;
        }
        return headers;
      }, {});
      return response.custom({
        body: registryResponse.body,
        statusCode: registryResponse.status,
        headers: {
          ...CACHE_CONTROL_10_MINUTES_HEADER,
          ...proxiedHeaders
        }
      });
    }
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.getFileHandler = getFileHandler;