"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentUpgradeStatus = void 0;
exports.getDownloadEstimate = getDownloadEstimate;
exports.getUpgradeStartDelay = getUpgradeStartDelay;
var _react = _interopRequireWildcard(require("react"));
var _i18nReact = require("@kbn/i18n-react");
var _eui = require("@elastic/eui");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns a user-friendly string for the estimated remaining time until the upgrade is scheduled.
 */
function getUpgradeStartDelay(scheduledAt) {
  const timeDiffMillis = Date.parse(scheduledAt || '') - Date.now();
  if (timeDiffMillis < 0) {
    // The scheduled time should not be in the past, this would indicate an issue.
    // We choose not to provide a time estimation rather than negative duration.
    return '';
  }
  if (timeDiffMillis < 15 * 6e4) {
    return ' The upgrade will start in less than 15 minutes.';
  }
  if (timeDiffMillis < 30 * 6e4) {
    return ' The upgrade will start in less than 30 minutes.';
  }
  if (timeDiffMillis < 60 * 6e4) {
    return ' The upgrade will start in less than 1 hour.';
  }
  return ` The upgrade will start in less than ${Math.ceil(timeDiffMillis / 36e5)} hours.`;
}
function getDownloadEstimate(metadata) {
  if (!metadata || metadata.download_percent === undefined && metadata.download_rate === undefined) {
    return '';
  }
  let tooltip = '';
  if (metadata.download_percent !== undefined) {
    tooltip = `${metadata.download_percent}%`;
  }
  if (metadata.download_rate !== undefined) {
    tooltip += ` at ${formatRate(metadata.download_rate)}`;
  }
  return ` (${tooltip.trim()})`;
}
const formatRate = downloadRate => {
  let i = 0;
  const byteUnits = [' Bps', ' kBps', ' MBps', ' GBps'];
  for (; i < byteUnits.length - 1; i++) {
    if (downloadRate < 1024) break;
    downloadRate = downloadRate / 1024;
  }
  return downloadRate.toFixed(1) + byteUnits[i];
};
function getStatusComponents(agentUpgradeDetails) {
  var _agentUpgradeDetails$, _agentUpgradeDetails$2;
  switch (agentUpgradeDetails === null || agentUpgradeDetails === void 0 ? void 0 : agentUpgradeDetails.state) {
    case 'UPG_REQUESTED':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "accent",
          iconType: "calendar"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeRequested",
          defaultMessage: "Upgrade requested"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeRequested",
          defaultMessage: "The agent has requested an upgrade."
        })
      };
    case 'UPG_SCHEDULED':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "accent",
          iconType: "clock"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeScheduled",
          defaultMessage: "Upgrade scheduled"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeScheduled",
          defaultMessage: "The agent has been instructed to upgrade.{upgradeStartDelay}",
          values: {
            upgradeStartDelay: getUpgradeStartDelay((_agentUpgradeDetails$ = agentUpgradeDetails.metadata) === null || _agentUpgradeDetails$ === void 0 ? void 0 : _agentUpgradeDetails$.scheduled_at)
          }
        })
      };
    case 'UPG_DOWNLOADING':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "accent",
          iconType: "download"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeDownloading",
          defaultMessage: "Upgrade downloading"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeDownloading",
          defaultMessage: "Downloading the new agent artifact version{downloadEstimate}.",
          values: {
            downloadEstimate: getDownloadEstimate(agentUpgradeDetails === null || agentUpgradeDetails === void 0 ? void 0 : agentUpgradeDetails.metadata)
          }
        })
      };
    case 'UPG_EXTRACTING':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "accent",
          iconType: "package"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeExtracting",
          defaultMessage: "Upgrade extracting"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeExtracting",
          defaultMessage: "The new agent artifact is extracting."
        })
      };
    case 'UPG_REPLACING':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "warning",
          iconType: "copy"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeReplacing",
          defaultMessage: "Upgrade replacing"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeReplacing",
          defaultMessage: "Replacing the agent artifact version."
        })
      };
    case 'UPG_RESTARTING':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "warning",
          iconType: "refresh"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeRestarting",
          defaultMessage: "Upgrade restarting"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeRestarting",
          defaultMessage: "The agent is restarting to apply the update."
        })
      };
    case 'UPG_WATCHING':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "warning",
          iconType: "inspect"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeMonitoring",
          defaultMessage: "Upgrade monitoring"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeMonitoring",
          defaultMessage: "Monitoring the new agent version for errors."
        })
      };
    case 'UPG_ROLLBACK':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "danger",
          iconType: "returnKey"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeRolledBack",
          defaultMessage: "Upgrade rolled back"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeRolledBack",
          defaultMessage: "Upgrade unsuccessful. Rolling back to previous version."
        })
      };
    case 'UPG_FAILED':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "danger",
          iconType: "error"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeFailed",
          defaultMessage: "Upgrade failed"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeFailed",
          defaultMessage: "Upgrade failed: {errorMsg}.",
          values: {
            errorMsg: agentUpgradeDetails === null || agentUpgradeDetails === void 0 ? void 0 : (_agentUpgradeDetails$2 = agentUpgradeDetails.metadata) === null || _agentUpgradeDetails$2 === void 0 ? void 0 : _agentUpgradeDetails$2.error_msg
          }
        })
      };
    default:
      return null;
  }
}
const AgentUpgradeStatus = ({
  isAgentUpgradable,
  agentUpgradeStartedAt,
  agentUpgradedAt,
  agentUpgradeDetails
}) => {
  const isAgentUpgrading = (0, _react.useMemo)(() => agentUpgradeStartedAt && !agentUpgradedAt, [agentUpgradeStartedAt, agentUpgradedAt]);
  const status = (0, _react.useMemo)(() => getStatusComponents(agentUpgradeDetails), [agentUpgradeDetails]);
  const minVersion = '8.12';
  if (isAgentUpgradable) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
      color: "hollow",
      iconType: "sortUp"
    }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.agentUpgradeStatusBadge.upgradeAvailable",
      defaultMessage: "Upgrade available"
    }));
  }
  if (agentUpgradeDetails && status) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
      gutterSize: "s",
      alignItems: "center",
      responsive: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, status.Badge), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
      type: "iInCircle",
      content: status.TooltipText,
      color: "subdued"
    })));
  }
  if (isAgentUpgrading) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
      type: "iInCircle",
      content: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeDetailsNotAvailable",
        defaultMessage: "Detailed upgrade status is available for Elastic Agents on version {minVersion} and higher.",
        values: {
          minVersion
        }
      }),
      color: "subdued"
    });
  }
  return null;
};
exports.AgentUpgradeStatus = AgentUpgradeStatus;