"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.apiRoutes = void 0;
exports.createFilesClient = createFilesClient;
var _api_routes = require("../../common/api_routes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * @internal
 */
const apiRoutes = {
  /**
   * Scoped to file kind
   */
  getCreateFileRoute: fileKind => `${_api_routes.FILES_API_BASE_PATH}/${fileKind}`,
  getUploadRoute: (fileKind, id) => `${_api_routes.FILES_API_BASE_PATH}/${fileKind}/${id}/blob`,
  getDownloadRoute: (fileKind, id, fileName) => `${_api_routes.FILES_API_BASE_PATH}/${fileKind}/${id}/blob${fileName ? '/' + fileName : ''}`,
  getUpdateRoute: (fileKind, id) => `${_api_routes.FILES_API_BASE_PATH}/${fileKind}/${id}`,
  getDeleteRoute: (fileKind, id) => `${_api_routes.FILES_API_BASE_PATH}/${fileKind}/${id}`,
  getListRoute: fileKind => `${_api_routes.FILES_API_BASE_PATH}/${fileKind}/list`,
  getByIdRoute: (fileKind, id) => `${_api_routes.FILES_API_BASE_PATH}/${fileKind}/${id}`,
  /**
   * Scope to file shares and file kind
   */
  getShareRoute: (fileKind, id) => `${_api_routes.FILES_SHARE_API_BASE_PATH}/${fileKind}/${id}`,
  getListSharesRoute: fileKind => `${_api_routes.FILES_SHARE_API_BASE_PATH}/${fileKind}`,
  /**
   * Public routes
   */
  getPublicDownloadRoute: fileName => `${_api_routes.FILES_PUBLIC_API_BASE_PATH}/blob${fileName ? '/' + fileName : ''}`,
  /**
   * Top-level routes
   */
  getFindRoute: () => `${_api_routes.API_BASE_PATH}/find`,
  getMetricsRoute: () => `${_api_routes.API_BASE_PATH}/metrics`,
  getBulkDeleteRoute: () => `${_api_routes.API_BASE_PATH}/blobs`
};

/**
 * Arguments to create a new {@link FileClient}.
 */

/**
 * Arguments to create a new {@link ScopedFilesClient}.
 */
exports.apiRoutes = apiRoutes;
const commonBodyHeaders = {
  headers: {
    'content-type': 'application/json'
  }
};
function createFilesClient({
  registry,
  http,
  fileKind: scopedFileKind
}) {
  const api = {
    bulkDelete: args => {
      return http.delete(apiRoutes.getBulkDeleteRoute(), {
        headers: commonBodyHeaders,
        body: JSON.stringify(args)
      });
    },
    create: ({
      kind,
      ...args
    }) => {
      return http.post(apiRoutes.getCreateFileRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind), {
        headers: commonBodyHeaders,
        body: JSON.stringify(args)
      });
    },
    delete: ({
      kind,
      ...args
    }) => {
      return http.delete(apiRoutes.getDeleteRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind, args.id));
    },
    download: ({
      kind,
      ...args
    }) => {
      return http.get(apiRoutes.getDownloadRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind, args.id, args.fileName), {
        headers: {
          Accept: '*/*'
        }
      });
    },
    getById: ({
      kind,
      ...args
    }) => {
      return http.get(apiRoutes.getByIdRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind, args.id));
    },
    list: ({
      kind,
      page,
      perPage,
      abortSignal,
      ...body
    } = {
      kind: ''
    }) => {
      return http.post(apiRoutes.getListRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind), {
        headers: commonBodyHeaders,
        query: {
          page,
          perPage
        },
        body: JSON.stringify(body),
        signal: abortSignal
      });
    },
    update: ({
      kind,
      id,
      ...body
    }) => {
      return http.patch(apiRoutes.getUpdateRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind, id), {
        headers: commonBodyHeaders,
        body: JSON.stringify(body)
      });
    },
    upload: ({
      kind,
      abortSignal,
      contentType,
      selfDestructOnAbort,
      ...args
    }) => {
      return http.put(apiRoutes.getUploadRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind, args.id), {
        query: {
          selfDestructOnAbort
        },
        headers: {
          'Content-Type': contentType !== null && contentType !== void 0 ? contentType : 'application/octet-stream'
        },
        signal: abortSignal,
        body: args.body
      });
    },
    getDownloadHref: ({
      fileKind: kind,
      id
    }) => {
      return `${http.basePath.prepend(apiRoutes.getDownloadRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind, id))}`;
    },
    share: ({
      kind,
      fileId,
      name,
      validUntil
    }) => {
      return http.post(apiRoutes.getShareRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind, fileId), {
        headers: commonBodyHeaders,
        body: JSON.stringify({
          name,
          validUntil
        })
      });
    },
    unshare: ({
      kind,
      id
    }) => {
      return http.delete(apiRoutes.getShareRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind, id));
    },
    getShare: ({
      kind,
      id
    }) => {
      return http.get(apiRoutes.getShareRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind, id));
    },
    listShares: ({
      kind,
      forFileId,
      page,
      perPage
    }) => {
      return http.get(apiRoutes.getListSharesRoute(scopedFileKind !== null && scopedFileKind !== void 0 ? scopedFileKind : kind), {
        query: {
          page,
          perPage,
          forFileId
        }
      });
    },
    find: ({
      page,
      perPage,
      ...filterArgs
    }) => {
      return http.post(apiRoutes.getFindRoute(), {
        query: {
          page,
          perPage
        },
        headers: commonBodyHeaders,
        body: JSON.stringify(filterArgs)
      });
    },
    getMetrics: () => {
      return http.get(apiRoutes.getMetricsRoute());
    },
    publicDownload: ({
      token,
      fileName
    }) => {
      return http.get(apiRoutes.getPublicDownloadRoute(fileName), {
        query: {
          token
        }
      });
    },
    getFileKind(id) {
      return registry.get(id);
    }
  };
  return api;
}