"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GeoJsonImporter = exports.GEOJSON_FILE_TYPES = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _i18n = require("@kbn/i18n");
var _json = require("@loaders.gl/json");
var _core = require("@loaders.gl/core");
var _abstract_geo_file_importer = require("../abstract_geo_file_importer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const GEOJSON_FILE_TYPES = ['.json', '.geojson'];
exports.GEOJSON_FILE_TYPES = GEOJSON_FILE_TYPES;
const SUPPORTED_CRS_LIST = ['EPSG:4326', 'urn:ogc:def:crs:OGC:1.3:CRS84'];
class GeoJsonImporter extends _abstract_geo_file_importer.AbstractGeoFileImporter {
  constructor(...args) {
    super(...args);
    (0, _defineProperty2.default)(this, "_iterator", void 0);
    (0, _defineProperty2.default)(this, "_prevBatchLastFeature", void 0);
  }
  async _readNext(prevTotalFeaturesRead, prevTotalBytesRead) {
    var _batch$container;
    let featureIndex = prevTotalFeaturesRead;
    const results = {
      bytesRead: 0,
      features: [],
      geometryTypesMap: new Map(),
      invalidFeatures: [],
      hasNext: true
    };
    const jsonOptions = {
      jsonpaths: ['$.features']
    };
    const jsonLoaderOptions = {
      // enabling metadata adds 3 additional batches to iterator output
      // 1) batchType: 'metadata' - ignored
      // 2) batchType: 'partial-result' - used to test for compatible crs, and fail early for incompatible crs
      // n) batchType: 'data' - unchanged by enabling metadata option
      // 3) batchType: 'final-result' - used to read top level feature when file is a single feature instead of feature collection
      metadata: true,
      json: jsonOptions
    };
    if (this._iterator === undefined) {
      this._iterator = (await (0, _core.loadInBatches)(this._getFile(), _json.JSONLoader, jsonLoaderOptions))[Symbol.asyncIterator]();
    }
    if (!this._getIsActive() || !this._iterator) {
      results.hasNext = false;
      return results;
    }
    const {
      value: batch,
      done
    } = await this._iterator.next();
    if (!this._getIsActive() || done || batch === undefined) {
      results.hasNext = false;
      return results;
    }
    if (batch.batchType === 'metadata') {
      return results;
    }

    // geojson only supports WGS 84 datum, with longitude and latitude units of decimal degrees.
    // https://datatracker.ietf.org/doc/html/rfc7946#section-4
    // Deprecated geojson specification supported crs
    // https://geojson.org/geojson-spec.html#named-crs
    // This importer only supports WGS 84 datum
    if (typeof ((_batch$container = batch.container) === null || _batch$container === void 0 ? void 0 : _batch$container.crs) === 'object') {
      var _crs$properties$name, _crs$properties;
      const crs = batch.container.crs;
      if ((crs === null || crs === void 0 ? void 0 : crs.type) === 'link' || (crs === null || crs === void 0 ? void 0 : crs.type) === 'name' && !SUPPORTED_CRS_LIST.includes((_crs$properties$name = crs === null || crs === void 0 ? void 0 : (_crs$properties = crs.properties) === null || _crs$properties === void 0 ? void 0 : _crs$properties.name) !== null && _crs$properties$name !== void 0 ? _crs$properties$name : '')) {
        throw new Error(_i18n.i18n.translate('xpack.fileUpload.geojsonImporter.unsupportedCrs', {
          defaultMessage: 'Unsupported coordinate reference system, expecting {supportedCrsList}',
          values: {
            supportedCrsList: SUPPORTED_CRS_LIST.join(', ')
          }
        }));
      }
    }
    if (batch.bytesUsed) {
      results.bytesRead = batch.bytesUsed - prevTotalBytesRead;
    }
    const features = this._prevBatchLastFeature ? [this._prevBatchLastFeature] : [];
    this._prevBatchLastFeature = undefined;
    const isLastBatch = batch.batchType === 'final-result';
    if (isLastBatch) {
      // Handle single feature geoJson
      if (featureIndex === 0 && features.length === 0) {
        if (batch.container) {
          features.push(batch.container);
        }
      }
    } else {
      if (batch.data) {
        features.push(...batch.data);
      }
    }
    for (let i = 0; i < features.length; i++) {
      const feature = features[i];
      if (!isLastBatch && i === features.length - 1) {
        // Do not process last feature until next batch is read, features on batch boundary may be incomplete.
        this._prevBatchLastFeature = feature;
        continue;
      }
      featureIndex++;
      if (!feature.geometry || !feature.geometry.type) {
        results.invalidFeatures.push({
          item: featureIndex,
          reason: _i18n.i18n.translate('xpack.fileUpload.geojsonImporter.noGeometry', {
            defaultMessage: 'Feature does not contain required field "geometry"'
          }),
          doc: feature
        });
      } else {
        if (!results.geometryTypesMap.has(feature.geometry.type)) {
          results.geometryTypesMap.set(feature.geometry.type, true);
        }
        results.features.push(feature);
      }
    }
    return results;
  }
}
exports.GeoJsonImporter = GeoJsonImporter;