"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SourceLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../shared/constants");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _kibana = require("../../../shared/kibana");
var _app_logic = require("../../app_logic");
var _routes = require("../../routes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SourceLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'workplace_search', 'source_logic'],
  actions: {
    setContentSource: contentSource => contentSource,
    onUpdateSourceName: name => name,
    onUpdateSummary: summary => summary,
    setSearchResults: searchResultsResponse => searchResultsResponse,
    setContentFilterValue: contentFilterValue => contentFilterValue,
    setActivePage: activePage => activePage,
    initializeSource: sourceId => ({
      sourceId
    }),
    initializeFederatedSummary: sourceId => ({
      sourceId
    }),
    initializeSourceSynchronization: sourceId => ({
      sourceId
    }),
    searchContentSourceDocuments: sourceId => ({
      sourceId
    }),
    updateContentSource: (sourceId, source) => ({
      sourceId,
      source
    }),
    updateContentSourceConfiguration: (sourceId, source) => ({
      sourceId,
      source
    }),
    removeContentSource: sourceId => ({
      sourceId
    }),
    resetSourceState: () => true,
    setButtonNotLoading: () => false,
    setStagedPrivateKey: stagedPrivateKey => stagedPrivateKey,
    setConfigurationUpdateButtonNotLoading: () => false,
    showDiagnosticDownloadButton: true
  },
  reducers: {
    contentSource: [{}, {
      setContentSource: (_, contentSource) => contentSource,
      onUpdateSourceName: (contentSource, name) => ({
        ...contentSource,
        name
      }),
      onUpdateSummary: (contentSource, summary) => ({
        ...contentSource,
        summary
      }),
      resetSourceState: () => ({})
    }],
    dataLoading: [true, {
      setContentSource: () => false,
      resetSourceState: () => true,
      removeContentSource: () => true
    }],
    buttonLoading: [false, {
      setButtonNotLoading: () => false,
      resetSourceState: () => false
    }],
    sectionLoading: [true, {
      searchContentSourceDocuments: () => true,
      setSearchResults: () => false
    }],
    diagnosticDownloadButtonVisible: [false, {
      showDiagnosticDownloadButton: () => true,
      initializeSource: () => false
    }],
    contentItems: [[], {
      setSearchResults: (_, {
        results
      }) => results
    }],
    contentMeta: [_constants.DEFAULT_META, {
      setActivePage: (state, activePage) => setPage(state, activePage),
      setContentFilterValue: state => setPage(state, _constants.DEFAULT_META.page.current),
      setSearchResults: (_, {
        meta
      }) => meta
    }],
    contentFilterValue: ['', {
      setContentFilterValue: (_, contentFilterValue) => contentFilterValue,
      resetSourceState: () => ''
    }],
    stagedPrivateKey: [null, {
      setStagedPrivateKey: (_, stagedPrivateKey) => stagedPrivateKey,
      setContentSource: () => null
    }],
    isConfigurationUpdateButtonLoading: [false, {
      updateContentSourceConfiguration: () => true,
      setConfigurationUpdateButtonNotLoading: () => false
    }]
  },
  listeners: ({
    actions,
    values
  }) => ({
    initializeSource: async ({
      sourceId
    }) => {
      const {
        isOrganization
      } = _app_logic.AppLogic.values;
      const route = isOrganization ? `/internal/workplace_search/org/sources/${sourceId}` : `/internal/workplace_search/account/sources/${sourceId}`;
      try {
        const response = await _http.HttpLogic.values.http.get(route);
        actions.setContentSource(response);
        if (response.isFederatedSource) {
          actions.initializeFederatedSummary(sourceId);
        }
        if (response.errors) {
          (0, _flash_messages.setErrorMessage)(response.errors);
          if (errorsHaveDiagnosticBundleString(response.errors)) {
            actions.showDiagnosticDownloadButton();
          }
        } else {
          (0, _flash_messages.clearFlashMessages)();
        }
      } catch (e) {
        var _e$response;
        if ((e === null || e === void 0 ? void 0 : (_e$response = e.response) === null || _e$response === void 0 ? void 0 : _e$response.status) === 404) {
          const redirect = isOrganization ? _routes.SOURCES_PATH : _routes.PRIVATE_SOURCES_PATH;
          _kibana.KibanaLogic.values.navigateToUrl(redirect);
          (0, _flash_messages.setErrorMessage)(_i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.sources.notFoundErrorMessage', {
            defaultMessage: 'Source not found.'
          }));
        } else {
          (0, _flash_messages.flashAPIErrors)(e);
        }
      }
    },
    initializeFederatedSummary: async ({
      sourceId
    }) => {
      const route = `/internal/workplace_search/account/sources/${sourceId}/federated_summary`;
      try {
        const response = await _http.HttpLogic.values.http.get(route);
        actions.onUpdateSummary(response.summary);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    searchContentSourceDocuments: async ({
      sourceId
    }, breakpoint) => {
      await breakpoint(300);
      const {
        isOrganization
      } = _app_logic.AppLogic.values;
      const route = isOrganization ? `/internal/workplace_search/org/sources/${sourceId}/documents` : `/internal/workplace_search/account/sources/${sourceId}/documents`;
      const {
        contentFilterValue: query,
        contentMeta: {
          page
        }
      } = values;
      try {
        const response = await _http.HttpLogic.values.http.post(route, {
          body: JSON.stringify({
            query,
            page
          })
        });
        actions.setSearchResults(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    updateContentSource: async ({
      sourceId,
      source
    }) => {
      const {
        isOrganization
      } = _app_logic.AppLogic.values;
      const route = isOrganization ? `/internal/workplace_search/org/sources/${sourceId}/settings` : `/internal/workplace_search/account/sources/${sourceId}/settings`;
      try {
        const response = await _http.HttpLogic.values.http.patch(route, {
          body: JSON.stringify({
            content_source: source
          })
        });
        if (source.name) {
          actions.onUpdateSourceName(response.name);
        }
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    updateContentSourceConfiguration: async ({
      sourceId,
      source
    }) => {
      const {
        isOrganization
      } = _app_logic.AppLogic.values;
      const route = isOrganization ? `/internal/workplace_search/org/sources/${sourceId}/settings` : `/internal/workplace_search/account/sources/${sourceId}/settings`;
      try {
        const response = await _http.HttpLogic.values.http.patch(route, {
          body: JSON.stringify({
            content_source: source
          })
        });
        actions.setContentSource(response);
        (0, _flash_messages.flashSuccessToast)('Content source configuration was updated.');
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      } finally {
        actions.setConfigurationUpdateButtonNotLoading();
      }
    },
    removeContentSource: async ({
      sourceId
    }) => {
      (0, _flash_messages.clearFlashMessages)();
      const {
        isOrganization
      } = _app_logic.AppLogic.values;
      const route = isOrganization ? `/internal/workplace_search/org/sources/${sourceId}` : `/internal/workplace_search/account/sources/${sourceId}`;
      try {
        const response = await _http.HttpLogic.values.http.delete(route);
        _kibana.KibanaLogic.values.navigateToUrl((0, _routes.getSourcesPath)(_routes.SOURCES_PATH, isOrganization));
        (0, _flash_messages.flashSuccessToast)(_i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.sources.flashMessages.contentSourceRemoved', {
          defaultMessage: 'Successfully deleted {sourceName}.',
          values: {
            sourceName: response.name
          }
        }));
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      } finally {
        actions.setButtonNotLoading();
      }
    },
    initializeSourceSynchronization: async ({
      sourceId
    }) => {
      const route = `/internal/workplace_search/org/sources/${sourceId}/sync`;
      try {
        await _http.HttpLogic.values.http.post(route);
        actions.initializeSource(sourceId);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    onUpdateSourceName: name => {
      (0, _flash_messages.flashSuccessToast)(_i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.sources.flashMessages.contentSourceNameChanged', {
        defaultMessage: 'Successfully changed name to {sourceName}.',
        values: {
          sourceName: name
        }
      }));
    },
    resetSourceState: () => {
      (0, _flash_messages.clearFlashMessages)();
    }
  })
});
exports.SourceLogic = SourceLogic;
const setPage = (state, page) => ({
  ...state,
  page: {
    ...state.page,
    current: page
  }
});
const errorsHaveDiagnosticBundleString = errors => {
  const ERROR_SUBSTRING = 'Check diagnostic bundle for details';
  return errors.find(e => e.includes(ERROR_SUBSTRING));
};