"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getE5MultilingualError = exports.E5MultilingualCalloutLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _api = require("../../../../../../../../common/types/api");
var _ml = require("../../../../../../../../common/types/ml");
var _handle_api_errors = require("../../../../../../shared/flash_messages/handle_api_errors");
var _create_e5_multilingual_model_api_logic = require("../../../../../api/ml_models/e5_multilingual/create_e5_multilingual_model_api_logic");
var _fetch_e5_multilingual_model_api_logic = require("../../../../../api/ml_models/e5_multilingual/fetch_e5_multilingual_model_api_logic");
var _start_e5_multilingual_model_api_logic = require("../../../../../api/ml_models/e5_multilingual/start_e5_multilingual_model_api_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FETCH_E5_MULTILINGUAL_MODEL_POLLING_DURATION = 5000; // 5 seconds
const FETCH_E5_MULTILINGUAL_MODEL_POLLING_DURATION_ON_FAILURE = 30000; // 30 seconds
const E5_MULTILINGUAL_MODEL_ID = '.multilingual-e5-small';
/**
 * Extracts the topmost error in precedence order (create > start > fetch).
 * @param createError
 * @param fetchError
 * @param startError
 * @returns the extracted error or null if there is no error
 */
const getE5MultilingualError = (createError, fetchError, startError) => {
  return createError !== undefined ? {
    title: _i18n.i18n.translate('xpack.enterpriseSearch.content.indices.pipelines.e5MultilingualCreateError.title', {
      defaultMessage: 'Error with E5 Multilingual deployment'
    }),
    message: (0, _handle_api_errors.getErrorsFromHttpResponse)(createError)[0]
  } : startError !== undefined ? {
    title: _i18n.i18n.translate('xpack.enterpriseSearch.content.indices.pipelines.e5MultilingualStartError.title', {
      defaultMessage: 'Error starting E5 Multilingual deployment'
    }),
    message: (0, _handle_api_errors.getErrorsFromHttpResponse)(startError)[0]
  } : fetchError !== undefined ? {
    title: _i18n.i18n.translate('xpack.enterpriseSearch.content.indices.pipelines.e5MultilingualFetchError.title', {
      defaultMessage: 'Error fetching E5 Multilingual model'
    }),
    message: (0, _handle_api_errors.getErrorsFromHttpResponse)(fetchError)[0]
  } : null;
};
exports.getE5MultilingualError = getE5MultilingualError;
const E5MultilingualCalloutLogic = (0, _kea.kea)({
  actions: {
    clearE5MultilingualModelPollingId: true,
    createE5MultilingualModelPollingTimeout: duration => ({
      duration
    }),
    setE5MultilingualModelPollingId: pollTimeoutId => ({
      pollTimeoutId
    }),
    startPollingE5MultilingualModel: true,
    stopPollingE5MultilingualModel: true,
    createE5MultilingualModel: true,
    fetchE5MultilingualModel: true,
    startE5MultilingualModel: true
  },
  connect: {
    actions: [_create_e5_multilingual_model_api_logic.CreateE5MultilingualModelApiLogic, ['makeRequest as createE5MultilingualModelMakeRequest', 'apiSuccess as createE5MultilingualModelSuccess', 'apiError as createE5MultilingualModelError'], _fetch_e5_multilingual_model_api_logic.FetchE5MultilingualModelApiLogic, ['makeRequest as fetchE5MultilingualModelMakeRequest', 'apiSuccess as fetchE5MultilingualModelSuccess', 'apiError as fetchE5MultilingualModelError'], _start_e5_multilingual_model_api_logic.StartE5MultilingualModelApiLogic, ['makeRequest as startE5MultilingualModelMakeRequest', 'apiSuccess as startE5MultilingualModelSuccess', 'apiError as startE5MultilingualModelError']],
    values: [_create_e5_multilingual_model_api_logic.CreateE5MultilingualModelApiLogic, ['data as createdE5MultilingualModel', 'status as createE5MultilingualModelStatus', 'error as createE5MultilingualModelError'], _fetch_e5_multilingual_model_api_logic.FetchE5MultilingualModelApiLogic, ['data as e5MultilingualModel', 'error as fetchE5MultilingualModelError'], _start_e5_multilingual_model_api_logic.StartE5MultilingualModelApiLogic, ['status as startE5MultilingualModelStatus', 'error as startE5MultilingualModelError']]
  },
  events: ({
    actions,
    values
  }) => ({
    afterMount: () => {
      actions.fetchE5MultilingualModel();
    },
    beforeUnmount: () => {
      if (values.e5MultilingualModelPollTimeoutId !== null) {
        actions.stopPollingE5MultilingualModel();
      }
    }
  }),
  listeners: ({
    actions,
    values
  }) => ({
    createE5MultilingualModel: () => actions.createE5MultilingualModelMakeRequest({
      modelId: E5_MULTILINGUAL_MODEL_ID
    }),
    fetchE5MultilingualModel: () => actions.fetchE5MultilingualModelMakeRequest({
      modelId: E5_MULTILINGUAL_MODEL_ID
    }),
    startE5MultilingualModel: () => actions.startE5MultilingualModelMakeRequest({
      modelId: E5_MULTILINGUAL_MODEL_ID
    }),
    createE5MultilingualModelPollingTimeout: ({
      duration
    }) => {
      if (values.e5MultilingualModelPollTimeoutId !== null) {
        clearTimeout(values.e5MultilingualModelPollTimeoutId);
      }
      const timeoutId = setTimeout(() => {
        actions.fetchE5MultilingualModel();
      }, duration);
      actions.setE5MultilingualModelPollingId(timeoutId);
    },
    createE5MultilingualModelSuccess: () => {
      actions.fetchE5MultilingualModel();
      actions.startPollingE5MultilingualModel();
    },
    fetchE5MultilingualModelError: () => {
      if (values.isPollingE5MultilingualModelActive) {
        actions.createE5MultilingualModelPollingTimeout(FETCH_E5_MULTILINGUAL_MODEL_POLLING_DURATION_ON_FAILURE);
      }
    },
    fetchE5MultilingualModelSuccess: data => {
      if ((data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Downloading) {
        if (!values.isPollingE5MultilingualModelActive) {
          actions.startPollingE5MultilingualModel();
        } else {
          actions.createE5MultilingualModelPollingTimeout(FETCH_E5_MULTILINGUAL_MODEL_POLLING_DURATION);
        }
      } else if ((data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Downloaded && values.isPollingE5MultilingualModelActive) {
        actions.stopPollingE5MultilingualModel();
      }
    },
    startPollingE5MultilingualModel: () => {
      if (values.e5MultilingualModelPollTimeoutId !== null) {
        clearTimeout(values.e5MultilingualModelPollTimeoutId);
      }
      actions.createE5MultilingualModelPollingTimeout(FETCH_E5_MULTILINGUAL_MODEL_POLLING_DURATION);
    },
    startE5MultilingualModelSuccess: () => {
      actions.fetchE5MultilingualModel();
    },
    stopPollingE5MultilingualModel: () => {
      if (values.e5MultilingualModelPollTimeoutId !== null) {
        clearTimeout(values.e5MultilingualModelPollTimeoutId);
        actions.clearE5MultilingualModelPollingId();
      }
    }
  }),
  path: ['enterprise_search', 'content', 'e5_multilingual_callout_logic'],
  reducers: {
    e5MultilingualModelPollTimeoutId: [null, {
      clearE5MultilingualModelPollingId: () => null,
      setE5MultilingualModelPollingId: (_, {
        pollTimeoutId
      }) => pollTimeoutId
    }]
  },
  selectors: ({
    selectors
  }) => ({
    isCreateButtonDisabled: [() => [selectors.createE5MultilingualModelStatus], status => status !== _api.Status.IDLE && status !== _api.Status.ERROR],
    isModelDownloadInProgress: [() => [selectors.e5MultilingualModel], data => (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Downloading],
    isModelDownloaded: [() => [selectors.e5MultilingualModel], data => (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Downloaded],
    isModelStarted: [() => [selectors.e5MultilingualModel], data => (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Starting || (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Started || (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.FullyAllocated],
    isPollingE5MultilingualModelActive: [() => [selectors.e5MultilingualModelPollTimeoutId], pollingTimeoutId => pollingTimeoutId !== null],
    e5MultilingualError: [() => [selectors.createE5MultilingualModelError, selectors.fetchE5MultilingualModelError, selectors.startE5MultilingualModelError], (createE5MultilingualError, fetchE5MultilingualError, startE5MultilingualError) => getE5MultilingualError(createE5MultilingualError, fetchE5MultilingualError, startE5MultilingualError)],
    isStartButtonDisabled: [() => [selectors.startE5MultilingualModelStatus], status => status !== _api.Status.IDLE && status !== _api.Status.ERROR],
    isModelRunningSingleThreaded: [() => [selectors.e5MultilingualModel], data =>
    // Running single threaded if model has max 1 deployment on 1 node with 1 thread
    (data === null || data === void 0 ? void 0 : data.targetAllocationCount) * (data === null || data === void 0 ? void 0 : data.threadsPerAllocation) <= 1]
  })
});
exports.E5MultilingualCalloutLogic = E5MultilingualCalloutLogic;