"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IndexViewLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _searchConnectors = require("@kbn/search-connectors");
var _flash_messages = require("../../../shared/flash_messages");
var _kibana = require("../../../shared/kibana");
var _start_access_control_sync_api_logic = require("../../api/connector/start_access_control_sync_api_logic");
var _start_incremental_sync_api_logic = require("../../api/connector/start_incremental_sync_api_logic");
var _start_sync_api_logic = require("../../api/connector/start_sync_api_logic");
var _update_connector_configuration_api_logic = require("../../api/connector/update_connector_configuration_api_logic");
var _cached_fetch_index_api_logic = require("../../api/index/cached_fetch_index_api_logic");
var _indices = require("../../utils/indices");
var _crawler_logic = require("./crawler/crawler_logic");
var _index_name_logic = require("./index_name_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const IndexViewLogic = (0, _kea.kea)({
  actions: {
    fetchIndex: true,
    recheckIndex: true,
    resetRecheckIndexLoading: true,
    startAccessControlSync: true,
    startIncrementalSync: true,
    startSync: true
  },
  connect: {
    actions: [_index_name_logic.IndexNameLogic, ['setIndexName'], _cached_fetch_index_api_logic.CachedFetchIndexApiLogic, ['apiError as fetchIndexApiError', 'apiReset as resetFetchIndexApi', 'apiSuccess as fetchIndexApiSuccess', 'makeRequest as makeFetchIndexRequest', 'startPolling as startFetchIndexPoll', 'stopPolling as stopFetchIndexPoll'], _update_connector_configuration_api_logic.ConnectorConfigurationApiLogic, ['apiSuccess as updateConfigurationApiSuccess'], _start_sync_api_logic.StartSyncApiLogic, ['apiSuccess as startSyncApiSuccess', 'makeRequest as makeStartSyncRequest'], _start_incremental_sync_api_logic.StartIncrementalSyncApiLogic, ['apiSuccess as startIncrementalSyncApiSuccess', 'makeRequest as makeStartIncrementalSyncRequest'], _start_access_control_sync_api_logic.StartAccessControlSyncApiLogic, ['apiSuccess as startAccessControlSyncApiSuccess', 'makeRequest as makeStartAccessControlSyncRequest'], _crawler_logic.CrawlerLogic, ['fetchCrawlerData']],
    values: [_index_name_logic.IndexNameLogic, ['indexName'], _cached_fetch_index_api_logic.CachedFetchIndexApiLogic, ['fetchIndexApiData', 'status as fetchIndexApiStatus', 'indexData', 'isInitialLoading']]
  },
  events: ({
    actions
  }) => ({
    beforeUnmount: () => {
      actions.stopFetchIndexPoll();
      actions.resetFetchIndexApi();
    }
  }),
  listeners: ({
    actions,
    values
  }) => ({
    fetchIndex: () => {
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      actions.makeFetchIndexRequest({
        indexName
      });
    },
    fetchIndexApiSuccess: index => {
      if ((0, _indices.isCrawlerIndex)(index) && index.name === values.indexName) {
        actions.fetchCrawlerData();
      }
      if (values.recheckIndexLoading) {
        actions.resetRecheckIndexLoading();
        (0, _flash_messages.flashSuccessToast)(_i18n.i18n.translate('xpack.enterpriseSearch.content.searchIndex.index.recheckSuccess.message', {
          defaultMessage: 'Your connector has been rechecked.'
        }));
      }
    },
    recheckIndex: () => actions.fetchIndex(),
    setIndexName: ({
      indexName
    }) => {
      actions.startFetchIndexPoll(indexName);
    },
    startAccessControlSync: () => {
      if ((0, _indices.isConnectorIndex)(values.fetchIndexApiData) && values.hasDocumentLevelSecurityFeature && _kibana.KibanaLogic.values.productFeatures.hasDocumentLevelSecurityEnabled) {
        actions.makeStartAccessControlSyncRequest({
          connectorId: values.fetchIndexApiData.connector.id
        });
      }
    },
    startIncrementalSync: () => {
      if ((0, _indices.isConnectorIndex)(values.fetchIndexApiData) && values.hasIncrementalSyncFeature && _kibana.KibanaLogic.values.productFeatures.hasIncrementalSyncEnabled) {
        actions.makeStartIncrementalSyncRequest({
          connectorId: values.fetchIndexApiData.connector.id
        });
      }
    },
    startSync: () => {
      if ((0, _indices.isConnectorIndex)(values.fetchIndexApiData)) {
        actions.makeStartSyncRequest({
          connectorId: values.fetchIndexApiData.connector.id
        });
      }
    },
    updateConfigurationApiSuccess: ({
      configuration
    }) => {
      if ((0, _indices.isConnectorIndex)(values.fetchIndexApiData)) {
        actions.fetchIndexApiSuccess({
          ...values.fetchIndexApiData,
          connector: {
            ...values.fetchIndexApiData.connector,
            configuration
          }
        });
      }
    }
  }),
  path: ['enterprise_search', 'content', 'index_view_logic'],
  reducers: {
    recheckIndexLoading: [false, {
      recheckIndex: () => true,
      resetRecheckIndexLoading: () => false
    }],
    syncTriggeredLocally: [false, {
      fetchIndexApiSuccess: () => false,
      startSyncApiSuccess: () => true
    }]
  },
  selectors: ({
    selectors
  }) => ({
    connector: [() => [selectors.indexData], index => index && ((0, _indices.isConnectorViewIndex)(index) || (0, _indices.isCrawlerIndex)(index)) ? index.connector : undefined],
    connectorError: [() => [selectors.connector], connector => connector === null || connector === void 0 ? void 0 : connector.error],
    connectorId: [() => [selectors.indexData], index => (0, _indices.isConnectorViewIndex)(index) ? index.connector.id : null],
    error: [() => [selectors.connector], connector => (connector === null || connector === void 0 ? void 0 : connector.error) || (connector === null || connector === void 0 ? void 0 : connector.last_sync_error) || null],
    hasAdvancedFilteringFeature: [() => [selectors.connector], connector => {
      var _connector$features$F, _connector$features$F2, _connector$features$F3;
      return connector !== null && connector !== void 0 && connector.features ? (_connector$features$F = (_connector$features$F2 = connector.features[_searchConnectors.FeatureName.SYNC_RULES]) === null || _connector$features$F2 === void 0 ? void 0 : (_connector$features$F3 = _connector$features$F2.advanced) === null || _connector$features$F3 === void 0 ? void 0 : _connector$features$F3.enabled) !== null && _connector$features$F !== void 0 ? _connector$features$F : connector.features[_searchConnectors.FeatureName.FILTERING_ADVANCED_CONFIG] : false;
    }],
    hasBasicFilteringFeature: [() => [selectors.connector], connector => {
      var _connector$features$F4, _connector$features$F5, _connector$features$F6;
      return connector !== null && connector !== void 0 && connector.features ? (_connector$features$F4 = (_connector$features$F5 = connector.features[_searchConnectors.FeatureName.SYNC_RULES]) === null || _connector$features$F5 === void 0 ? void 0 : (_connector$features$F6 = _connector$features$F5.basic) === null || _connector$features$F6 === void 0 ? void 0 : _connector$features$F6.enabled) !== null && _connector$features$F4 !== void 0 ? _connector$features$F4 : connector.features[_searchConnectors.FeatureName.FILTERING_RULES] : false;
    }],
    hasDocumentLevelSecurityFeature: [() => [selectors.connector], connector => {
      var _connector$features, _connector$features$F7;
      return (connector === null || connector === void 0 ? void 0 : (_connector$features = connector.features) === null || _connector$features === void 0 ? void 0 : (_connector$features$F7 = _connector$features[_searchConnectors.FeatureName.DOCUMENT_LEVEL_SECURITY]) === null || _connector$features$F7 === void 0 ? void 0 : _connector$features$F7.enabled) || false;
    }],
    hasFilteringFeature: [() => [selectors.hasAdvancedFilteringFeature, selectors.hasBasicFilteringFeature], (advancedFeature, basicFeature) => advancedFeature || basicFeature],
    hasIncrementalSyncFeature: [() => [selectors.connector], connector => {
      var _connector$features2, _connector$features2$;
      return (connector === null || connector === void 0 ? void 0 : (_connector$features2 = connector.features) === null || _connector$features2 === void 0 ? void 0 : (_connector$features2$ = _connector$features2[_searchConnectors.FeatureName.INCREMENTAL_SYNC]) === null || _connector$features2$ === void 0 ? void 0 : _connector$features2$.enabled) || false;
    }],
    htmlExtraction: [() => [selectors.connector], connector => {
      var _connector$configurat, _connector$configurat2;
      return (_connector$configurat = connector === null || connector === void 0 ? void 0 : (_connector$configurat2 = connector.configuration.extract_full_html) === null || _connector$configurat2 === void 0 ? void 0 : _connector$configurat2.value) !== null && _connector$configurat !== void 0 ? _connector$configurat : undefined;
    }],
    index: [() => [selectors.indexData], data => data ? (0, _indices.indexToViewIndex)(data) : undefined],
    ingestionMethod: [() => [selectors.indexData], data => (0, _indices.getIngestionMethod)(data)],
    ingestionStatus: [() => [selectors.indexData], data => (0, _indices.getIngestionStatus)(data)],
    isCanceling: [() => [selectors.syncStatus], syncStatus => syncStatus === _searchConnectors.SyncStatus.CANCELING],
    isConnectorIndex: [() => [selectors.indexData], data => (0, _indices.isConnectorIndex)(data)],
    isHiddenIndex: [() => [selectors.indexData], data => {
      var _data$name;
      return (data === null || data === void 0 ? void 0 : data.hidden) || ((_data$name = data === null || data === void 0 ? void 0 : data.name) !== null && _data$name !== void 0 ? _data$name : '').startsWith('.');
    }],
    isSyncing: [() => [selectors.indexData, selectors.syncStatus], (indexData, syncStatus) => (indexData === null || indexData === void 0 ? void 0 : indexData.has_in_progress_syncs) || syncStatus === _searchConnectors.SyncStatus.IN_PROGRESS],
    isWaitingForSync: [() => [selectors.indexData, selectors.syncTriggeredLocally], (indexData, syncTriggeredLocally) => (indexData === null || indexData === void 0 ? void 0 : indexData.has_pending_syncs) || syncTriggeredLocally || false],
    lastUpdated: [() => [selectors.fetchIndexApiData], data => (0, _indices.getLastUpdated)(data)],
    pipelineData: [() => [selectors.connector], connector => {
      var _connector$pipeline;
      return (_connector$pipeline = connector === null || connector === void 0 ? void 0 : connector.pipeline) !== null && _connector$pipeline !== void 0 ? _connector$pipeline : undefined;
    }],
    syncStatus: [() => [selectors.fetchIndexApiData], data => {
      var _data$connector$last_, _data$connector;
      return (_data$connector$last_ = data === null || data === void 0 ? void 0 : (_data$connector = data.connector) === null || _data$connector === void 0 ? void 0 : _data$connector.last_sync_status) !== null && _data$connector$last_ !== void 0 ? _data$connector$last_ : null;
    }]
  })
});
exports.IndexViewLogic = IndexViewLogic;