"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AttributeService = exports.ATTRIBUTE_SERVICE_KEY = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _react = _interopRequireDefault(require("react"));
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _public = require("@kbn/saved-objects-plugin/public");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * The attribute service is a shared, generic service that embeddables can use to provide the functionality
 * required to fulfill the requirements of the ReferenceOrValueEmbeddable interface. The attribute_service
 * can also be used as a higher level wrapper to transform an embeddable input shape that references a saved object
 * into an embeddable input shape that contains that saved object's attributes by value.
 */
const ATTRIBUTE_SERVICE_KEY = 'attributes';
exports.ATTRIBUTE_SERVICE_KEY = ATTRIBUTE_SERVICE_KEY;
class AttributeService {
  constructor(type, toasts, options, getEmbeddableFactory) {
    (0, _defineProperty2.default)(this, "embeddableFactory", void 0);
    (0, _defineProperty2.default)(this, "inputIsRefType", input => {
      return (0, _.isSavedObjectEmbeddableInput)(input);
    });
    (0, _defineProperty2.default)(this, "getInputAsValueType", async input => {
      if (!this.inputIsRefType(input)) {
        return input;
      }
      const {
        attributes
      } = await this.unwrapAttributes(input);
      const {
        savedObjectId,
        ...originalInputToPropagate
      } = input;
      return {
        ...originalInputToPropagate,
        // by value visualizations should not have default titles and/or descriptions
        ...{
          attributes: (0, _lodash.omit)(attributes, ['title', 'description'])
        }
      };
    });
    (0, _defineProperty2.default)(this, "getInputAsRefType", async (input, saveOptions) => {
      if (this.inputIsRefType(input)) {
        return input;
      }
      return new Promise((resolve, reject) => {
        const onSave = async props => {
          await this.options.checkForDuplicateTitle(props);
          try {
            const newAttributes = {
              ...input[ATTRIBUTE_SERVICE_KEY]
            };
            newAttributes.title = props.newTitle;
            const wrappedInput = await this.wrapAttributes(newAttributes, true);
            // Remove unneeded attributes from the original input. Note that the original panel title
            // is removed in favour of the new attributes title
            const newInput = (0, _lodash.omit)(input, [ATTRIBUTE_SERVICE_KEY, 'title']);

            // Combine input and wrapped input to preserve any passed in explicit Input
            resolve({
              ...newInput,
              ...wrappedInput
            });
            return {
              id: wrappedInput.savedObjectId
            };
          } catch (error) {
            reject(error);
            return {
              error
            };
          }
        };
        if (saveOptions && saveOptions.showSaveModal) {
          (0, _public.showSaveModal)( /*#__PURE__*/_react.default.createElement(_public.SavedObjectSaveModal, {
            onSave: onSave,
            onClose: () => {},
            title: (0, _lodash.get)(saveOptions, 'saveModalTitle', input[ATTRIBUTE_SERVICE_KEY].title),
            showCopyOnSave: false,
            objectType: this.embeddableFactory ? this.embeddableFactory.getDisplayName() : this.type,
            showDescription: false
          }));
        }
      });
    });
    this.type = type;
    this.toasts = toasts;
    this.options = options;
    if (getEmbeddableFactory) {
      const factory = getEmbeddableFactory(this.type);
      if (!factory) {
        throw new _.EmbeddableFactoryNotFoundError(this.type);
      }
      this.embeddableFactory = factory;
    }
  }
  async defaultUnwrapMethod(input) {
    return Promise.resolve({
      attributes: {
        ...input
      }
    });
  }
  async unwrapAttributes(input) {
    if (this.inputIsRefType(input)) {
      return this.options.unwrapMethod ? await this.options.unwrapMethod(input.savedObjectId) : await this.defaultUnwrapMethod(input);
    }
    return {
      attributes: input[ATTRIBUTE_SERVICE_KEY]
    };
  }
  async wrapAttributes(newAttributes, useRefType, input) {
    const originalInput = input ? input : {};
    const savedObjectId = input && this.inputIsRefType(input) ? input.savedObjectId : undefined;
    if (!useRefType) {
      return {
        [ATTRIBUTE_SERVICE_KEY]: newAttributes
      };
    }
    try {
      const savedItem = await this.options.saveMethod(newAttributes, savedObjectId);
      if ('id' in savedItem) {
        return {
          ...originalInput,
          savedObjectId: savedItem.id
        };
      }
      return {
        ...originalInput
      };
    } catch (error) {
      this.toasts.addDanger({
        title: _i18n.i18n.translate('embeddableApi.attributeService.saveToLibraryError', {
          defaultMessage: `An error occurred while saving. Error: {errorMessage}`,
          values: {
            errorMessage: error.message
          }
        }),
        'data-test-subj': 'attributeServiceSaveFailure'
      });
      return Promise.reject({
        error
      });
    }
  }
}
exports.AttributeService = AttributeService;