"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performEvaluation = void 0;
var _evaluation = require("langchain/evaluation");
var _schema = require("langchain/schema");
var _fp = require("lodash/fp");
var _callbacks = require("langchain/callbacks");
var _utils = require("./utils");
var _utils2 = require("../../routes/evaluate/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Evaluates a dataset based on an evaluation rubric. Takes dataset of input/reference pairs,
 * and fetches the output (prediction) of the input against the provided agent executors.
 * Then evaluates all three using the provided evaluation rubric.
 */
const performEvaluation = async ({
  agentExecutorEvaluators,
  dataset,
  evaluationId,
  evaluatorModel,
  evaluationPrompt,
  evaluationType,
  maxConcurrency = 1,
  logger,
  runName = 'default-run-name'
}) => {
  const startTime = new Date().getTime();
  const evaluationResults = [];
  const predictionRequests = dataset.flatMap(({
    input,
    reference,
    id: exampleId
  }) => agentExecutorEvaluators.map(({
    agentEvaluator: agent,
    metadata: {
      connectorName,
      runName: agentRunName
    }
  }) => ({
    connectorName,
    input,
    reference,
    exampleId,
    request: () => (0, _utils.callAgentWithRetry)({
      agent,
      exampleId,
      messages: [new _schema.HumanMessage(input)],
      logger
    }),
    runName: agentRunName
  })));
  const requestChunks = (0, _fp.chunk)(maxConcurrency, predictionRequests);
  const totalChunks = requestChunks.length;
  logger.info(`Total prediction requests: ${predictionRequests.length}`);
  logger.info(`Chunk size (maxConcurrency): ${maxConcurrency}`);
  logger.info(`Total chunks: ${totalChunks}`);
  logger.info('Fetching predictions...');
  while (requestChunks.length) {
    var _requestChunks$shift;
    const chunk = (_requestChunks$shift = requestChunks.shift()) !== null && _requestChunks$shift !== void 0 ? _requestChunks$shift : [];
    const chunkNumber = totalChunks - requestChunks.length;
    logger.info(`Prediction request chunk: ${chunkNumber} of ${totalChunks}`);
    logger.debug(chunk);

    // Note, order is kept between chunk and dataset, and is preserved w/ Promise.allSettled
    const chunkResults = await Promise.allSettled(chunk.map(r => r.request()));
    logger.info(`Prediction request chunk ${chunkNumber} response:\n${JSON.stringify(chunkResults)}`);
    chunkResults.forEach((response, chunkResultIndex) => evaluationResults.push({
      '@timestamp': new Date().toISOString(),
      connectorName: chunk[chunkResultIndex].connectorName,
      input: chunk[chunkResultIndex].input,
      inputExampleId: chunk[chunkResultIndex].exampleId,
      reference: chunk[chunkResultIndex].reference,
      evaluationId,
      evaluation: {},
      prediction: (0, _utils.getMessageFromLangChainResponse)(response),
      predictionResponse: response,
      runName: chunk[chunkResultIndex].runName
    }));
  }
  logger.info(`Prediction results:\n${JSON.stringify(evaluationResults)}`);
  if (evaluatorModel == null) {
    const endTime = new Date().getTime();
    const evaluationSummary = {
      evaluationId,
      '@timestamp': new Date().toISOString(),
      evaluationStart: startTime,
      evaluationEnd: endTime,
      evaluationDuration: endTime - startTime,
      runName,
      totalAgents: agentExecutorEvaluators.length,
      totalInput: dataset.length,
      totalRequests: predictionRequests.length
    };
    logger.info(`Final results:\n${JSON.stringify(evaluationResults)}`);
    return {
      evaluationResults,
      evaluationSummary
    };
  }

  // Continue with actual evaluation if expected
  logger.info('Performing evaluation....');
  logger.info(`Evaluation model: ${evaluatorModel._llmType()}`);
  if (evaluationType === 'correctness') {
    logger.info('Evaluation type: correctness');
    const evaluator = await (0, _evaluation.loadEvaluator)('labeled_criteria', {
      criteria: 'correctness',
      llm: evaluatorModel
    });
    for (const result of evaluationResults) {
      const {
        input,
        inputExampleId: exampleId,
        prediction,
        reference
      } = result;
      // Create an eval tracer so eval traces end up in the right project (runName in this instance as to correlate
      // with the test run), don't supply `exampleID` as that results in a new Dataset `Test` run being created and
      // polluting the `predictions` that ran above
      const evalTracer = new _callbacks.LangChainTracer({
        projectName: runName
      });
      // Create RunCollector for uploading evals to LangSmith, no TS variant for `EvaluatorCallbackHandler` or
      // `run_on_dataset` w/ eval config, so using `RunCollectorCallbackHandler` and then uploading manually via
      // client.createFeedback()
      // See: https://github.com/langchain-ai/langsmith-sdk/blob/18449e5848d85ac0a320f320c37f454f949de1e1/js/src/client.ts#L1249-L1256
      const runCollector = new _callbacks.RunCollectorCallbackHandler({
        exampleId
      });
      const evaluation = await evaluator.evaluateStrings({
        input,
        prediction,
        reference
      }, {
        callbacks: [...((0, _utils2.isLangSmithEnabled)() ? [evalTracer, runCollector] : [])],
        tags: ['security-assistant-evaluation']
      });
      result.evaluation = evaluation;

      // Write to LangSmith
      if ((0, _utils2.isLangSmithEnabled)()) {
        const langSmithLink = await (0, _utils2.writeLangSmithFeedback)(runCollector.tracedRuns[0], evaluationId, logger);
        result.langSmithLink = langSmithLink;
      }
    }
  } else if (evaluationType === 'esql-validator') {
    logger.info('Evaluation type: esql-validator');
    // TODO: Implement esql-validator here
  } else if (evaluationType === 'custom') {
    logger.info('Evaluation type: custom');
    // TODO: Implement custom evaluation here
    // const llm = new ChatOpenAI({ temperature: 0, tags: ["my-llm-tag"] });
    // const prompt = PromptTemplate.fromTemplate("Say {input}");
    // const chain = prompt.pipe(llm).withConfig( { tags: ["my-bash-tag", "another-tag"] });
    // await chain.invoke({ input: "Hello, World!"}, { tags: ["shared-tags"] });
  }

  const endTime = new Date().getTime();
  const evaluationSummary = {
    evaluationId,
    '@timestamp': new Date().toISOString(),
    evaluationStart: startTime,
    evaluationEnd: endTime,
    evaluationDuration: endTime - startTime,
    runName,
    totalAgents: agentExecutorEvaluators.length,
    totalInput: dataset.length,
    totalRequests: predictionRequests.length
  };
  logger.info(`Final results:\n${JSON.stringify(evaluationResults)}`);
  return {
    evaluationResults,
    evaluationSummary
  };
};
exports.performEvaluation = performEvaluation;