"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callAgentExecutor = exports.DEFAULT_AGENT_EXECUTOR_ID = void 0;
var _agents = require("langchain/agents");
var _chains = require("langchain/chains");
var _memory = require("langchain/memory");
var _elasticsearch_store = require("../elasticsearch_store/elasticsearch_store");
var _actions_client_llm = require("../llm/actions_client_llm");
var _constants = require("../../../routes/knowledge_base/constants");
var _with_assistant_span = require("../tracers/with_assistant_span");
var _apm_tracer = require("../tracers/apm_tracer");
var _tools = require("../tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_AGENT_EXECUTOR_ID = 'Elastic AI Assistant Agent Executor';

/**
 * The default agent executor used by the Elastic AI Assistant. Main agent/chain that wraps the ActionsClientLlm,
 * sets up a conversation BufferMemory from chat history, and registers tools like the ESQLKnowledgeBaseTool.
 *
 */
exports.DEFAULT_AGENT_EXECUTOR_ID = DEFAULT_AGENT_EXECUTOR_ID;
const callAgentExecutor = async ({
  actions,
  alertsIndexPattern,
  allow,
  allowReplacement,
  assistantLangChain,
  connectorId,
  elserId,
  esClient,
  kbResource,
  langChainMessages,
  llmType,
  logger,
  onNewReplacements,
  replacements,
  request,
  size,
  traceOptions
}) => {
  var _traceOptions$project;
  const llm = new _actions_client_llm.ActionsClientLlm({
    actions,
    connectorId,
    request,
    llmType,
    logger
  });
  const pastMessages = langChainMessages.slice(0, -1); // all but the last message
  const latestMessage = langChainMessages.slice(-1); // the last message

  const memory = new _memory.BufferMemory({
    chatHistory: new _memory.ChatMessageHistory(pastMessages),
    memoryKey: 'chat_history',
    // this is the key expected by https://github.com/langchain-ai/langchainjs/blob/a13a8969345b0f149c1ca4a120d63508b06c52a5/langchain/src/agents/initialize.ts#L166
    inputKey: 'input',
    outputKey: 'output',
    returnMessages: true
  });

  // ELSER backed ElasticsearchStore for Knowledge Base
  const esStore = new _elasticsearch_store.ElasticsearchStore(esClient, _constants.KNOWLEDGE_BASE_INDEX_PATTERN, logger, elserId, kbResource);
  const modelExists = await esStore.isModelInstalled();

  // Create a chain that uses the ELSER backed ElasticsearchStore, override k=10 for esql query generation for now
  const chain = _chains.RetrievalQAChain.fromLLM(llm, esStore.asRetriever(10));
  const tools = (0, _tools.getApplicableTools)({
    allow,
    allowReplacement,
    alertsIndexPattern,
    assistantLangChain,
    chain,
    esClient,
    modelExists,
    onNewReplacements,
    replacements,
    request,
    size
  });
  logger.debug(`applicable tools: ${JSON.stringify(tools.map(t => t.name).join(', '), null, 2)}`);
  const executor = await (0, _agents.initializeAgentExecutorWithOptions)(tools, llm, {
    agentType: 'chat-conversational-react-description',
    memory,
    verbose: false
  });

  // Sets up tracer for tracing executions to APM. See x-pack/plugins/elastic_assistant/server/lib/langchain/tracers/README.mdx
  // If LangSmith env vars are set, executions will be traced there as well. See https://docs.smith.langchain.com/tracing
  const apmTracer = new _apm_tracer.APMTracer({
    projectName: (_traceOptions$project = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.projectName) !== null && _traceOptions$project !== void 0 ? _traceOptions$project : 'default'
  }, logger);
  let traceData;

  // Wrap executor call with an APM span for instrumentation
  await (0, _with_assistant_span.withAssistantSpan)(DEFAULT_AGENT_EXECUTOR_ID, async span => {
    var _span$transaction, _traceOptions$tracers, _traceOptions$tags;
    if ((span === null || span === void 0 ? void 0 : (_span$transaction = span.transaction) === null || _span$transaction === void 0 ? void 0 : _span$transaction.ids['transaction.id']) != null && (span === null || span === void 0 ? void 0 : span.ids['trace.id']) != null) {
      traceData = {
        // Transactions ID since this span is the parent
        transaction_id: span.transaction.ids['transaction.id'],
        trace_id: span.ids['trace.id']
      };
      span.addLabels({
        evaluationId: traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.evaluationId
      });
    }
    return executor.call({
      input: latestMessage[0].content
    }, {
      callbacks: [apmTracer, ...((_traceOptions$tracers = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tracers) !== null && _traceOptions$tracers !== void 0 ? _traceOptions$tracers : [])],
      runName: DEFAULT_AGENT_EXECUTOR_ID,
      tags: (_traceOptions$tags = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tags) !== null && _traceOptions$tags !== void 0 ? _traceOptions$tags : []
    });
  });
  return {
    connector_id: connectorId,
    data: llm.getActionResultData(),
    // the response from the actions framework
    trace_data: traceData,
    replacements,
    status: 'ok'
  };
};
exports.callAgentExecutor = callAgentExecutor;