"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.swapReferencesRoute = exports.swapReference = void 0;
var _configSchema = require("@kbn/config-schema");
var _handle_errors = require("./util/handle_errors");
var _constants = require("../../constants");
var _constants2 = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const swapReference = async ({
  dataViewsService,
  usageCollection,
  counterName,
  id
}) => {
  usageCollection === null || usageCollection === void 0 ? void 0 : usageCollection.incrementCounter({
    counterName
  });
  return dataViewsService.get(id);
};
exports.swapReference = swapReference;
const idSchema = _configSchema.schema.string();
const swapReferencesRoute = ({
  previewRoute
}) => (router, getStartServices, usageCollection) => {
  const path = previewRoute ? `${_constants.DATA_VIEW_SWAP_REFERENCES_PATH}/_preview` : _constants.DATA_VIEW_SWAP_REFERENCES_PATH;
  router.versioned.post({
    path,
    access: 'public'
  }).addVersion({
    version: _constants.INITIAL_REST_VERSION,
    validate: {
      request: {
        body: _configSchema.schema.object({
          fromId: idSchema,
          fromType: _configSchema.schema.maybe(_configSchema.schema.string()),
          toId: idSchema,
          forId: _configSchema.schema.maybe(_configSchema.schema.oneOf([idSchema, _configSchema.schema.arrayOf(idSchema)])),
          forType: _configSchema.schema.maybe(_configSchema.schema.string()),
          delete: _configSchema.schema.maybe(_configSchema.schema.boolean())
        })
      },
      response: {
        200: {
          body: _configSchema.schema.object({
            result: _configSchema.schema.arrayOf(_configSchema.schema.object({
              id: idSchema,
              type: _configSchema.schema.string()
            })),
            deleteStatus: _configSchema.schema.maybe(_configSchema.schema.object({
              remainingRefs: _configSchema.schema.number(),
              deletePerformed: _configSchema.schema.boolean()
            }))
          })
        }
      }
    }
  }, router.handleLegacyErrors((0, _handle_errors.handleErrors)(async (ctx, req, res) => {
    const savedObjectsClient = (await ctx.core).savedObjects.client;
    const [core] = await getStartServices();
    const types = core.savedObjects.getTypeRegistry().getAllTypes();
    const type = req.body.fromType || _constants2.DATA_VIEW_SAVED_OBJECT_TYPE;
    const searchId = !Array.isArray(req.body.forId) && req.body.forId !== undefined ? [req.body.forId] : req.body.forId;
    usageCollection === null || usageCollection === void 0 ? void 0 : usageCollection.incrementCounter({
      counterName: 'swap_references'
    });

    // verify 'to' object actually exists
    try {
      await savedObjectsClient.get(type, req.body.toId);
    } catch (e) {
      throw new Error(`Could not find object with type ${type} and id ${req.body.toId}`);
    }

    // assemble search params
    const findParams = {
      type: types.map(t => t.name),
      hasReference: {
        type,
        id: req.body.fromId
      }
    };
    if (req.body.forType) {
      findParams.type = [req.body.forType];
    }
    const {
      saved_objects: savedObjects
    } = await savedObjectsClient.find(findParams);
    const filteredSavedObjects = searchId ? savedObjects.filter(so => searchId === null || searchId === void 0 ? void 0 : searchId.includes(so.id)) : savedObjects;

    // create summary of affected objects
    const resultSummary = filteredSavedObjects.map(savedObject => ({
      id: savedObject.id,
      type: savedObject.type
    }));
    const body = {
      result: resultSummary
    };

    // bail if preview
    if (previewRoute) {
      return res.ok({
        headers: {
          'content-type': 'application/json'
        },
        body
      });
    }

    // iterate over list and update references
    for (const savedObject of filteredSavedObjects) {
      const updatedRefs = savedObject.references.map(ref => {
        if (ref.type === type && ref.id === req.body.fromId) {
          return {
            ...ref,
            id: req.body.toId
          };
        } else {
          return ref;
        }
      });
      await savedObjectsClient.update(savedObject.type, savedObject.id, {}, {
        references: updatedRefs
      });
    }
    if (req.body.delete) {
      const verifyNoMoreRefs = await savedObjectsClient.find(findParams);
      if (verifyNoMoreRefs.total > 0) {
        body.deleteStatus = {
          remainingRefs: verifyNoMoreRefs.total,
          deletePerformed: false
        };
      } else {
        await savedObjectsClient.delete(type, req.body.fromId, {
          refresh: 'wait_for'
        });
        body.deleteStatus = {
          remainingRefs: verifyNoMoreRefs.total,
          deletePerformed: true
        };
      }
    }
    return res.ok({
      headers: {
        'content-type': 'application/json'
      },
      body
    });
  })));
};
exports.swapReferencesRoute = swapReferencesRoute;