"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FilterManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _rxjs = require("rxjs");
var _esQuery = require("@kbn/es-query");
var _sort_filters = require("./lib/sort_filters");
var _map_and_flatten_filters = require("./lib/map_and_flatten_filters");
var _common = require("../../../common");
var _persistable_state = require("../../../common/query/filters/persistable_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class FilterManager {
  constructor(uiSettings) {
    (0, _defineProperty2.default)(this, "filters", []);
    (0, _defineProperty2.default)(this, "updated$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "fetch$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "uiSettings", void 0);
    (0, _defineProperty2.default)(this, "extract", _persistable_state.extract);
    (0, _defineProperty2.default)(this, "inject", _persistable_state.inject);
    (0, _defineProperty2.default)(this, "telemetry", _persistable_state.telemetry);
    (0, _defineProperty2.default)(this, "getAllMigrations", _persistable_state.getAllMigrations);
    this.uiSettings = uiSettings;
  }
  mergeIncomingFilters(partitionedFilters) {
    const globalFilters = partitionedFilters.globalFilters;
    const appFilters = partitionedFilters.appFilters;

    // existing globalFilters should be mutated by appFilters
    // ignore original appFilters which are already inside globalFilters
    const cleanedAppFilters = [];
    _lodash.default.each(appFilters, function (filter, i) {
      const match = _lodash.default.find(globalFilters, function (globalFilter) {
        return (0, _esQuery.compareFilters)(globalFilter, filter);
      });

      // no match, do continue with app filter
      if (!match) {
        return cleanedAppFilters.push(filter);
      }

      // matching filter in globalState, update global and don't add from appState
      _lodash.default.assignIn(match.meta, filter.meta);
    });
    return FilterManager.mergeFilters(cleanedAppFilters, globalFilters);
  }
  static mergeFilters(appFilters, globalFilters) {
    return (0, _esQuery.uniqFilters)(appFilters.reverse().concat(globalFilters.reverse())).reverse();
  }
  static partitionFilters(filters) {
    const [globalFilters, appFilters] = _lodash.default.partition(filters, _esQuery.isFilterPinned);
    return {
      globalFilters,
      appFilters
    };
  }
  handleStateUpdate(newFilters) {
    newFilters.sort(_sort_filters.sortFilters);
    const filtersUpdated = !(0, _esQuery.compareFilters)(this.filters, newFilters, _esQuery.COMPARE_ALL_OPTIONS);
    const updatedOnlyDisabledFilters = (0, _esQuery.onlyDisabledFiltersChanged)(newFilters, this.filters);
    this.filters = newFilters;
    if (filtersUpdated) {
      this.updated$.next();
      if (!updatedOnlyDisabledFilters) {
        this.fetch$.next();
      }
    }
  }

  /* Getters */

  getFilters() {
    return _lodash.default.cloneDeep(this.filters);
  }
  getAppFilters() {
    const {
      appFilters
    } = this.getPartitionedFilters();
    return appFilters;
  }
  getGlobalFilters() {
    const {
      globalFilters
    } = this.getPartitionedFilters();
    return globalFilters;
  }
  getPartitionedFilters() {
    return FilterManager.partitionFilters(this.getFilters());
  }
  getUpdates$() {
    return this.updated$.asObservable();
  }
  getFetches$() {
    return this.fetch$.asObservable();
  }

  /* Setters */

  addFilters(filters, pinFilterStatus = this.uiSettings.get(_common.UI_SETTINGS.FILTERS_PINNED_BY_DEFAULT)) {
    if (!Array.isArray(filters)) {
      filters = [filters];
    }
    if (filters.length === 0) {
      return;
    }
    const store = pinFilterStatus ? _esQuery.FilterStateStore.GLOBAL_STATE : _esQuery.FilterStateStore.APP_STATE;
    FilterManager.setFiltersStore(filters, store);
    const mappedFilters = (0, _map_and_flatten_filters.mapAndFlattenFilters)(filters);

    // This is where we add new filters to the correct place (app \ global)
    const newPartitionedFilters = FilterManager.partitionFilters(mappedFilters);
    const currentFilters = this.getPartitionedFilters();
    currentFilters.appFilters.push(...newPartitionedFilters.appFilters);
    currentFilters.globalFilters.push(...newPartitionedFilters.globalFilters);
    const newFilters = this.mergeIncomingFilters(currentFilters);
    this.handleStateUpdate(newFilters);
  }
  setFilters(newFilters, pinFilterStatus = this.uiSettings.get(_common.UI_SETTINGS.FILTERS_PINNED_BY_DEFAULT)) {
    const store = pinFilterStatus ? _esQuery.FilterStateStore.GLOBAL_STATE : _esQuery.FilterStateStore.APP_STATE;
    FilterManager.setFiltersStore(newFilters, store);
    const mappedFilters = (0, _map_and_flatten_filters.mapAndFlattenFilters)(newFilters);
    const newPartitionedFilters = FilterManager.partitionFilters(mappedFilters);
    const mergedFilters = this.mergeIncomingFilters(newPartitionedFilters);
    this.handleStateUpdate(mergedFilters);
  }

  /**
   * Sets new global filters and leaves app filters untouched,
   * Removes app filters for which there is a duplicate within new global filters
   * @param newGlobalFilters
   */
  setGlobalFilters(newGlobalFilters) {
    newGlobalFilters = (0, _map_and_flatten_filters.mapAndFlattenFilters)(newGlobalFilters);
    FilterManager.setFiltersStore(newGlobalFilters, _esQuery.FilterStateStore.GLOBAL_STATE, true);
    const {
      appFilters
    } = this.getPartitionedFilters();
    const newFilters = this.mergeIncomingFilters({
      appFilters,
      globalFilters: newGlobalFilters
    });
    this.handleStateUpdate(newFilters);
  }

  /**
   * Sets new app filters and leaves global filters untouched,
   * Removes app filters for which there is a duplicate within new global filters
   * @param newAppFilters
   */
  setAppFilters(newAppFilters) {
    newAppFilters = (0, _map_and_flatten_filters.mapAndFlattenFilters)(newAppFilters);
    FilterManager.setFiltersStore(newAppFilters, _esQuery.FilterStateStore.APP_STATE, true);
    const {
      globalFilters
    } = this.getPartitionedFilters();
    const newFilters = this.mergeIncomingFilters({
      globalFilters,
      appFilters: newAppFilters
    });
    this.handleStateUpdate(newFilters);
  }
  removeFilter(filter) {
    const filterIndex = _lodash.default.findIndex(this.filters, item => {
      return _lodash.default.isEqual(item.meta, filter.meta) && _lodash.default.isEqual(item.query, filter.query);
    });
    if (filterIndex >= 0) {
      const newFilters = _lodash.default.cloneDeep(this.filters);
      newFilters.splice(filterIndex, 1);
      this.handleStateUpdate(newFilters);
    }
  }
  removeAll() {
    this.setFilters([]);
  }
  static setFiltersStore(filters, store, shouldOverrideStore = false) {
    _lodash.default.map(filters, filter => {
      // Override status only for filters that didn't have state in the first place.
      // or if shouldOverrideStore is explicitly true
      if (shouldOverrideStore || filter.$state === undefined) {
        filter.$state = {
          store
        };
      }
    });
  }
}
exports.FilterManager = FilterManager;