"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.serializeControlGroupInput = exports.saveDashboardState = exports.convertTimeToUTCString = void 0;
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _common = require("@kbn/controls-plugin/common");
var _esQuery = require("@kbn/es-query");
var _public = require("@kbn/data-plugin/public");
var _common2 = require("../../../../common");
var _dashboard_constants = require("../../../dashboard_constants");
var _dashboard_versioning = require("./dashboard_versioning");
var _dashboard_container = require("../../../dashboard_container");
var _dashboard_panel_converters = require("../../../../common/lib/dashboard_panel_converters");
var _dashboard_content_management_service = require("../dashboard_content_management_service");
var _dashboard_container_strings = require("../../../dashboard_container/_dashboard_container_strings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const serializeControlGroupInput = controlGroupInput => {
  // only save to saved object if control group is not default
  if (!controlGroupInput || (0, _common.persistableControlGroupInputIsEqual)(controlGroupInput, (0, _common.getDefaultControlGroupInput)())) {
    return undefined;
  }
  return (0, _common.controlGroupInputToRawControlGroupAttributes)(controlGroupInput);
};
exports.serializeControlGroupInput = serializeControlGroupInput;
const convertTimeToUTCString = time => {
  if ((0, _moment.default)(time).isValid()) {
    return (0, _moment.default)(time).utc().format('YYYY-MM-DDTHH:mm:ss.SSS[Z]');
  } else {
    // If it's not a valid moment date, then it should be a string representing a relative time
    // like 'now' or 'now-15m'.
    return time;
  }
};
exports.convertTimeToUTCString = convertTimeToUTCString;
const saveDashboardState = async ({
  data,
  embeddable,
  lastSavedId,
  saveOptions,
  currentState,
  dashboardBackup,
  contentManagement,
  savedObjectsTagging,
  notifications: {
    toasts
  }
}) => {
  const {
    search: dataSearchService,
    query: {
      timefilter: {
        timefilter
      }
    }
  } = data;
  const {
    tags,
    query,
    title,
    filters,
    version,
    timeRestore,
    description,
    // Dashboard options
    useMargins,
    syncColors,
    syncCursor,
    syncTooltips,
    hidePanelTitles
  } = currentState;
  let {
    panels,
    controlGroupInput
  } = currentState;
  if (saveOptions.saveAsCopy) {
    panels = (0, _dashboard_panel_converters.generateNewPanelIds)(panels);
    controlGroupInput = (0, _common.generateNewControlIds)(controlGroupInput);
  }

  /**
   * Stringify filters and query into search source JSON
   */
  const {
    searchSourceJSON,
    searchSourceReferences
  } = await (async () => {
    const searchSource = await dataSearchService.searchSource.create();
    searchSource.setField('filter',
    // save only unpinned filters
    filters.filter(filter => !(0, _esQuery.isFilterPinned)(filter)));
    searchSource.setField('query', query);
    const rawSearchSourceFields = searchSource.getSerializedFields();
    const [fields, references] = (0, _public.extractSearchSourceReferences)(rawSearchSourceFields);
    return {
      searchSourceReferences: references,
      searchSourceJSON: JSON.stringify(fields)
    };
  })();

  /**
   * Stringify options and panels
   */
  const optionsJSON = JSON.stringify({
    useMargins,
    syncColors,
    syncCursor,
    syncTooltips,
    hidePanelTitles
  });
  const panelsJSON = JSON.stringify((0, _common2.convertPanelMapToSavedPanels)(panels, true));

  /**
   * Parse global time filter settings
   */
  const {
    from,
    to
  } = timefilter.getTime();
  const timeFrom = timeRestore ? convertTimeToUTCString(from) : undefined;
  const timeTo = timeRestore ? convertTimeToUTCString(to) : undefined;
  const refreshInterval = timeRestore ? (0, _lodash.pick)(timefilter.getRefreshInterval(), ['display', 'pause', 'section', 'value']) : undefined;
  const rawDashboardAttributes = {
    version: (0, _dashboard_versioning.convertDashboardVersionToNumber)(version !== null && version !== void 0 ? version : _dashboard_container.LATEST_DASHBOARD_CONTAINER_VERSION),
    controlGroupInput: serializeControlGroupInput(controlGroupInput),
    kibanaSavedObjectMeta: {
      searchSourceJSON
    },
    description: description !== null && description !== void 0 ? description : '',
    refreshInterval,
    timeRestore,
    optionsJSON,
    panelsJSON,
    timeFrom,
    title,
    timeTo
  };

  /**
   * Extract references from raw attributes and tags into the references array.
   */
  const {
    attributes,
    references: dashboardReferences
  } = (0, _common2.extractReferences)({
    attributes: rawDashboardAttributes,
    references: searchSourceReferences
  }, {
    embeddablePersistableStateService: embeddable
  });
  const references = savedObjectsTagging.updateTagsReferences ? savedObjectsTagging.updateTagsReferences(dashboardReferences, tags) : dashboardReferences;

  /**
   * Save the saved object using the content management
   */
  const idToSaveTo = saveOptions.saveAsCopy ? undefined : lastSavedId;
  try {
    const result = await contentManagement.client.create({
      contentTypeId: _dashboard_constants.DASHBOARD_CONTENT_ID,
      data: attributes,
      options: {
        id: idToSaveTo,
        references,
        overwrite: true
      }
    });
    const newId = result.item.id;
    if (newId) {
      toasts.addSuccess({
        title: _dashboard_container_strings.dashboardSaveToastStrings.getSuccessString(currentState.title),
        className: 'eui-textBreakWord',
        'data-test-subj': 'saveDashboardSuccess'
      });

      /**
       * If the dashboard id has been changed, redirect to the new ID to keep the url param in sync.
       */
      if (newId !== lastSavedId) {
        dashboardBackup.clearState(lastSavedId);
        return {
          redirectRequired: true,
          id: newId
        };
      } else {
        _dashboard_content_management_service.dashboardContentManagementCache.deleteDashboard(newId); // something changed in an existing dashboard, so delete it from the cache so that it can be re-fetched
      }
    }

    return {
      id: newId
    };
  } catch (error) {
    toasts.addDanger({
      title: _dashboard_container_strings.dashboardSaveToastStrings.getFailureString(currentState.title, error.message),
      'data-test-subj': 'saveDashboardFailure'
    });
    return {
      error
    };
  }
};
exports.saveDashboardState = saveDashboardState;