"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConversionTransforms = getConversionTransforms;
exports.getCoreTransforms = getCoreTransforms;
exports.getReferenceTransforms = getReferenceTransforms;
var _lodash = require("lodash");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _migrations = require("./migrations");
var _types = require("./types");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Returns all available core transforms for all object types.
 */
function getCoreTransforms({
  type,
  log
}) {
  return Object.entries(_migrations.migrations).map(([version, transform]) => ({
    version,
    deferred: !(0, _lodash.isFunction)(transform) && !!transform.deferred,
    transform: (0, _utils.convertMigrationFunction)(version, type, transform, log),
    transformType: _types.TransformType.Core
  }));
}

/**
 * Returns all applicable conversion transforms for a given object type.
 */
function getConversionTransforms(type) {
  const {
    convertToMultiNamespaceTypeVersion
  } = type;
  if (!convertToMultiNamespaceTypeVersion) {
    return [];
  }
  return [{
    version: convertToMultiNamespaceTypeVersion,
    transform: convertNamespaceType,
    transformType: _types.TransformType.Convert
  }];
}

/**
 * Returns all applicable reference transforms for all object types.
 */
function getReferenceTransforms(typeRegistry) {
  const transformMap = typeRegistry.getAllTypes().filter(type => type.convertToMultiNamespaceTypeVersion).reduce((acc, {
    convertToMultiNamespaceTypeVersion: version,
    name
  }) => {
    var _acc$get;
    const types = (_acc$get = acc.get(version)) !== null && _acc$get !== void 0 ? _acc$get : new Set();
    return acc.set(version, types.add(name));
  }, new Map());
  return Array.from(transformMap, ([version, types]) => ({
    version,
    transform: doc => {
      const {
        namespace,
        references
      } = doc;
      if (namespace && references !== null && references !== void 0 && references.length) {
        return {
          transformedDoc: {
            ...doc,
            references: references.map(({
              type,
              id,
              ...attrs
            }) => ({
              ...attrs,
              type,
              id: types.has(type) ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.getConvertedObjectId(namespace, type, id) : id
            }))
          },
          additionalDocs: []
        };
      }
      return {
        transformedDoc: doc,
        additionalDocs: []
      };
    },
    transformType: _types.TransformType.Reference
  }));
}

/**
 * Converts a single-namespace object to a multi-namespace object. This primarily entails removing the `namespace` field and adding the
 * `namespaces` field.
 *
 * If the object does not exist in the default namespace (undefined), its ID is also regenerated, and an "originId" is added to preserve
 * legacy import/copy behavior.
 */
function convertNamespaceType(doc) {
  const {
    namespace,
    ...otherAttrs
  } = doc;
  const additionalDocs = [];
  if (namespace == null && otherAttrs.namespaces) {
    return {
      additionalDocs,
      transformedDoc: otherAttrs
    };
  }

  // If this object exists in the default namespace, return it with the appropriate `namespaces` field without changing its ID.
  if (namespace === undefined) {
    return {
      transformedDoc: {
        ...otherAttrs,
        namespaces: [_coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING]
      },
      additionalDocs
    };
  }
  const {
    id: originId,
    type
  } = otherAttrs;
  const id = _coreSavedObjectsUtilsServer.SavedObjectsUtils.getConvertedObjectId(namespace, type, originId);
  const legacyUrlAlias = {
    id: `${namespace}:${type}:${originId}`,
    type: _coreSavedObjectsBaseServerInternal.LEGACY_URL_ALIAS_TYPE,
    attributes: {
      // NOTE TO MAINTAINERS: If a saved object migration is added in `src/core/server/saved_objects/object_types/registration.ts`, these
      // values must be updated accordingly. That's because a user can upgrade Kibana from 7.17 to 8.x, and any defined migrations will not
      // be applied to aliases that are created during the conversion process.
      sourceId: originId,
      targetNamespace: namespace,
      targetType: type,
      targetId: id,
      purpose: 'savedObjectConversion'
    }
  };
  additionalDocs.push(legacyUrlAlias);
  return {
    transformedDoc: {
      ...otherAttrs,
      id,
      originId,
      namespaces: [namespace]
    },
    additionalDocs
  };
}