"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mergeGlobalContext = mergeGlobalContext;
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Takes a flattened object of {@link GlobalContext} and applies it to the
 * provided {@link LogMeta}.
 *
 * @remarks
 * The provided `LogMeta` takes precedence over the `GlobalContext`;
 * if duplicate keys are found, the `GlobalContext` will be overridden.
 *
 * @example
 * ```ts
 * const meta: LogMeta = {
 *   a: { b: false },
 *   d: 'hi',
 * };
 * const context: GlobalContext = {
 *   'a.b': true,
 *   c: [1, 2, 3],
 * };
 *
 * mergeGlobalContext(context, meta);
 * // {
 * //   a: { b: false },
 * //   c: [1, 2, 3],
 * //   d: 'hi',
 * // }
 * ```
 *
 * @internal
 */
function mergeGlobalContext(globalContext, meta) {
  if (!meta && Object.keys(globalContext).length === 0) {
    return;
  }
  const mergedMeta = meta !== null && meta !== void 0 ? meta : {};
  for (const [path, data] of Object.entries(globalContext)) {
    if (!(0, _lodash.has)(mergedMeta, path)) {
      (0, _saferLodashSet.set)(mergedMeta, path, data);
    }
  }
  return mergedMeta;
}