"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertValidationFunction = exports.Type = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _errors = require("../errors");
var _references = require("../references");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Options for dealing with unknown keys:
 * - allow: unknown keys will be permitted
 * - ignore: unknown keys will not fail validation, but will be stripped out
 * - forbid (default): unknown keys will fail validation
 */

const convertValidationFunction = validate => {
  return (value, {
    error
  }) => {
    let validationResultMessage;
    try {
      validationResultMessage = validate(value);
    } catch (e) {
      validationResultMessage = e.message || e;
    }
    if (typeof validationResultMessage === 'string') {
      return error('any.custom', {
        message: validationResultMessage
      });
    }
    return value;
  };
};
exports.convertValidationFunction = convertValidationFunction;
class Type {
  // This is just to enable the `TypeOf` helper, and because TypeScript would
  // fail if it wasn't initialized we use a "trick" to which basically just
  // sets the value to `null` while still keeping the type.

  // used for the `isConfigSchema` typeguard

  /**
   * Internal "schema" backed by Joi.
   * @type {Schema}
   */

  constructor(schema, options = {}) {
    (0, _defineProperty2.default)(this, "type", null);
    (0, _defineProperty2.default)(this, "__isKbnConfigSchemaType", true);
    (0, _defineProperty2.default)(this, "internalSchema", void 0);
    if (options.defaultValue !== undefined) {
      schema = schema.optional();

      // If default value is a function, then we must provide description for it.
      if (typeof options.defaultValue === 'function') {
        schema = schema.default(options.defaultValue);
      } else {
        schema = schema.default(_references.Reference.isReference(options.defaultValue) ? options.defaultValue.getSchema() : options.defaultValue);
      }
    }
    if (options.validate) {
      schema = schema.custom(convertValidationFunction(options.validate));
    }

    // Attach generic error handler only if it hasn't been attached yet since
    // only the last error handler is counted.
    if (schema.$_getFlag('error') === undefined) {
      schema = schema.error(([error]) => this.onError(error));
    }
    this.internalSchema = schema;
  }
  extendsDeep(newOptions) {
    return this;
  }
  validate(value, context = {}, namespace) {
    const {
      value: validatedValue,
      error
    } = this.internalSchema.validate(value, {
      context,
      presence: 'required'
    });
    if (error) {
      throw new _errors.ValidationError(error, namespace);
    }
    return validatedValue;
  }

  /**
   * @internal
   */
  getSchema() {
    return this.internalSchema;
  }
  getSchemaStructure() {
    return recursiveGetSchemaStructure(this.internalSchema);
  }
  handleError(type, context, path) {
    return undefined;
  }
  onError(error) {
    if (error instanceof _errors.SchemaTypeError) {
      return error;
    }
    const {
      local,
      code,
      path,
      value
    } = error;
    const convertedPath = path.map(entry => entry.toString());
    const context = {
      ...local,
      value
    };
    const errorHandleResult = this.handleError(code, context, convertedPath);
    if (errorHandleResult instanceof _errors.SchemaTypeError) {
      return errorHandleResult;
    }

    // If error handler just defines error message, then wrap it into proper
    // `SchemaTypeError` instance.
    if (typeof errorHandleResult === 'string') {
      return new _errors.SchemaTypeError(errorHandleResult, convertedPath);
    }

    // If error is produced by the custom validator, just extract source message
    // from context and wrap it into `SchemaTypeError` instance.
    if (code === 'any.custom' && context.message) {
      return new _errors.SchemaTypeError(context.message, convertedPath);
    }

    // `message` is only initialized once `toString` has been called (...)
    // see https://github.com/sideway/joi/blob/master/lib/errors.js
    const message = error.toString();
    return new _errors.SchemaTypeError(message || code, convertedPath);
  }
}
exports.Type = Type;
function recursiveGetSchemaStructure(internalSchema, path = []) {
  const array = [];
  // Note: we are relying on Joi internals to obtain the schema structure (recursive keys).
  // This is not ideal, but it works for now and we only need it for some integration test assertions.
  // If it breaks in the future, we'll need to update our tests.
  for (const [key, val] of internalSchema._ids._byKey.entries()) {
    array.push(...recursiveGetSchemaStructure(val.schema, [...path, key]));
  }
  if (!array.length) {
    var _internalSchema$type;
    array.push({
      path,
      type: (_internalSchema$type = internalSchema.type) !== null && _internalSchema$type !== void 0 ? _internalSchema$type : 'unknown'
    });
  }
  return array;
}