"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enable = enable;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _server = require("@kbn/task-manager-plugin/server");
var _lib = require("../../lib");
var _authorization = require("../../authorization");
var _retry_if_conflicts = require("../../lib/retry_if_conflicts");
var _audit_events = require("../common/audit_events");
var _lib2 = require("../lib");
var _get_schedule_frequency = require("../../application/rule/methods/get_schedule_frequency");
var _common = require("../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function enable(context, {
  id
}) {
  return await (0, _retry_if_conflicts.retryIfConflicts)(context.logger, `rulesClient.enable('${id}')`, async () => await enableWithOCC(context, {
    id
  }));
}
async function enableWithOCC(context, {
  id
}) {
  var _context$auditLogger2;
  let existingApiKey = null;
  let attributes;
  let version;
  let references;
  try {
    const decryptedAlert = await context.encryptedSavedObjectsClient.getDecryptedAsInternalUser('alert', id, {
      namespace: context.namespace
    });
    existingApiKey = decryptedAlert.attributes.apiKey;
    attributes = decryptedAlert.attributes;
    version = decryptedAlert.version;
    references = decryptedAlert.references;
  } catch (e) {
    context.logger.error(`enable(): Failed to load API key of alert ${id}: ${e.message}`);
    // Still attempt to load the attributes and version using SOC
    const alert = await context.unsecuredSavedObjectsClient.get('alert', id);
    attributes = alert.attributes;
    version = alert.version;
    references = alert.references;
  }
  const validationPayload = await (0, _get_schedule_frequency.validateScheduleLimit)({
    context,
    updatedInterval: attributes.schedule.interval
  });
  if (validationPayload) {
    throw _boom.default.badRequest((0, _common.getRuleCircuitBreakerErrorMessage)({
      name: attributes.name,
      interval: validationPayload.interval,
      intervalAvailable: validationPayload.intervalAvailable,
      action: 'enable'
    }));
  }
  try {
    await context.authorization.ensureAuthorized({
      ruleTypeId: attributes.alertTypeId,
      consumer: attributes.consumer,
      operation: _authorization.WriteOperations.Enable,
      entity: _authorization.AlertingAuthorizationEntity.Rule
    });
    if (attributes.actions.length) {
      await context.actionsAuthorization.ensureAuthorized({
        operation: 'execute'
      });
    }
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.ENABLE,
      savedObject: {
        type: 'alert',
        id
      },
      error
    }));
    throw error;
  }
  (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
    action: _audit_events.RuleAuditAction.ENABLE,
    outcome: 'unknown',
    savedObject: {
      type: 'alert',
      id
    }
  }));
  context.ruleTypeRegistry.ensureRuleTypeEnabled(attributes.alertTypeId);
  if (attributes.enabled === false) {
    const migratedActions = await (0, _lib2.migrateLegacyActions)(context, {
      ruleId: id,
      actions: attributes.actions,
      references,
      attributes
    });
    const username = await context.getUserName();
    const now = new Date();
    const schedule = attributes.schedule;
    const updateAttributes = (0, _lib2.updateMeta)(context, {
      ...attributes,
      ...(!existingApiKey && (await (0, _lib2.createNewAPIKeySet)(context, {
        id: attributes.alertTypeId,
        ruleName: attributes.name,
        username,
        shouldUpdateApiKey: true
      }))),
      ...(attributes.monitoring && {
        monitoring: (0, _lib.resetMonitoringLastRun)(attributes.monitoring)
      }),
      nextRun: (0, _lib.getNextRun)({
        interval: schedule.interval
      }),
      enabled: true,
      updatedBy: username,
      updatedAt: now.toISOString(),
      executionStatus: {
        status: 'pending',
        lastDuration: 0,
        lastExecutionDate: now.toISOString(),
        error: null,
        warning: null
      }
    });
    try {
      // to mitigate AAD issues(actions property is not used for encrypting API key in partial SO update)
      // we call create with overwrite=true
      if (migratedActions.hasLegacyActions) {
        await context.unsecuredSavedObjectsClient.create('alert', {
          ...updateAttributes,
          actions: migratedActions.resultedActions,
          throttle: undefined,
          notifyWhen: undefined
        }, {
          id,
          overwrite: true,
          version,
          references: migratedActions.resultedReferences
        });
      } else {
        await context.unsecuredSavedObjectsClient.update('alert', id, updateAttributes, {
          version
        });
      }
    } catch (e) {
      throw e;
    }
  }
  let scheduledTaskIdToCreate = null;
  if (attributes.scheduledTaskId) {
    // If scheduledTaskId defined in rule SO, make sure it exists
    try {
      const task = await context.taskManager.get(attributes.scheduledTaskId);

      // Check whether task status is unrecognized. If so, we want to delete
      // this task and create a fresh one
      if (task.status === _server.TaskStatus.Unrecognized) {
        await context.taskManager.removeIfExists(attributes.scheduledTaskId);
        scheduledTaskIdToCreate = id;
      }
    } catch (err) {
      scheduledTaskIdToCreate = id;
    }
  } else {
    // If scheduledTaskId doesn't exist in rule SO, set it to rule ID
    scheduledTaskIdToCreate = id;
  }
  if (scheduledTaskIdToCreate) {
    // Schedule the task if it doesn't exist
    const scheduledTask = await (0, _lib2.scheduleTask)(context, {
      id,
      consumer: attributes.consumer,
      ruleTypeId: attributes.alertTypeId,
      schedule: attributes.schedule,
      throwOnConflict: false
    });
    await context.unsecuredSavedObjectsClient.update('alert', id, {
      scheduledTaskId: scheduledTask.id
    });
  } else {
    // Task exists so set enabled to true
    await context.taskManager.bulkEnable([attributes.scheduledTaskId]);
  }
}