"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearCurrentActiveSnooze = clearCurrentActiveSnooze;
exports.clearCurrentActiveSnoozeAttributes = clearCurrentActiveSnoozeAttributes;
exports.clearScheduledSnoozesAttributesById = clearScheduledSnoozesAttributesById;
exports.clearScheduledSnoozesById = clearScheduledSnoozesById;
exports.clearUnscheduledSnooze = clearUnscheduledSnooze;
exports.clearUnscheduledSnoozeAttributes = clearUnscheduledSnoozeAttributes;
exports.getBulkSnooze = getBulkSnooze;
exports.getBulkUnsnooze = getBulkUnsnooze;
exports.getSnoozeAttributes = getSnoozeAttributes;
exports.getUnsnoozeAttributes = getUnsnoozeAttributes;
exports.verifySnoozeAttributeScheduleLimit = verifySnoozeAttributeScheduleLimit;
exports.verifySnoozeScheduleLimit = verifySnoozeScheduleLimit;
var _i18n = require("@kbn/i18n");
var _is_rule_snoozed = require("../../lib/is_rule_snoozed");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getSnoozeAttributes(attributes, snoozeSchedule) {
  // If duration is -1, instead mute all
  const {
    id: snoozeId,
    duration
  } = snoozeSchedule;
  if (duration === -1) {
    return {
      muteAll: true,
      snoozeSchedule: clearUnscheduledSnoozeAttributes(attributes)
    };
  }
  return {
    snoozeSchedule: (snoozeId ? clearScheduledSnoozesAttributesById(attributes, [snoozeId]) : clearUnscheduledSnoozeAttributes(attributes)).concat(snoozeSchedule),
    muteAll: false
  };
}
function getBulkSnooze(rule, snoozeSchedule) {
  // If duration is -1, instead mute all
  const {
    id: snoozeId,
    duration
  } = snoozeSchedule;
  if (duration === -1) {
    return {
      muteAll: true,
      snoozeSchedule: clearUnscheduledSnooze(rule)
    };
  }

  // Bulk adding snooze schedule, don't touch the existing snooze/indefinite snooze
  if (snoozeId) {
    const existingSnoozeSchedules = rule.snoozeSchedule || [];
    return {
      muteAll: rule.muteAll,
      snoozeSchedule: [...existingSnoozeSchedules, snoozeSchedule]
    };
  }

  // Bulk snoozing, don't touch the existing snooze schedules
  return {
    muteAll: false,
    snoozeSchedule: [...(clearUnscheduledSnooze(rule) || []), snoozeSchedule]
  };
}
function getUnsnoozeAttributes(attributes, scheduleIds) {
  const snoozeSchedule = scheduleIds ? clearScheduledSnoozesAttributesById(attributes, scheduleIds) : clearCurrentActiveSnoozeAttributes(attributes);
  return {
    snoozeSchedule,
    ...(!scheduleIds ? {
      muteAll: false
    } : {})
  };
}
function getBulkUnsnooze(rule, scheduleIds) {
  // Bulk removing snooze schedules, don't touch the current snooze/indefinite snooze
  if (scheduleIds) {
    var _rule$snoozeSchedule;
    const newSchedules = clearScheduledSnoozesById(rule, scheduleIds);
    // Unscheduled snooze is also known as snooze now
    const unscheduledSnooze = ((_rule$snoozeSchedule = rule.snoozeSchedule) === null || _rule$snoozeSchedule === void 0 ? void 0 : _rule$snoozeSchedule.filter(s => typeof s.id === 'undefined')) || [];
    return {
      snoozeSchedule: [...unscheduledSnooze, ...newSchedules],
      muteAll: rule.muteAll
    };
  }

  // Bulk unsnoozing, don't touch current snooze schedules that are NOT active
  return {
    snoozeSchedule: clearCurrentActiveSnooze(rule),
    muteAll: false
  };
}
function clearUnscheduledSnoozeAttributes(attributes) {
  // Clear any snoozes that have no ID property. These are "simple" snoozes created with the quick UI, e.g. snooze for 3 days starting now
  return attributes.snoozeSchedule ? attributes.snoozeSchedule.filter(s => typeof s.id !== 'undefined') : [];
}
function clearUnscheduledSnooze(rule) {
  return rule.snoozeSchedule ? rule.snoozeSchedule.filter(s => typeof s.id !== 'undefined') : [];
}
function clearScheduledSnoozesAttributesById(attributes, ids) {
  return attributes.snoozeSchedule ? attributes.snoozeSchedule.filter(s => s.id && !ids.includes(s.id)) : [];
}
function clearScheduledSnoozesById(rule, ids) {
  return rule.snoozeSchedule ? rule.snoozeSchedule.filter(s => s.id && !ids.includes(s.id)) : [];
}
function clearCurrentActiveSnoozeAttributes(attributes) {
  var _activeSnoozes$map;
  // First attempt to cancel a simple (unscheduled) snooze
  const clearedUnscheduledSnoozes = clearUnscheduledSnoozeAttributes(attributes);
  // Now clear any scheduled snoozes that are currently active and never recur
  const activeSnoozes = (0, _is_rule_snoozed.getActiveScheduledSnoozes)(attributes);
  const activeSnoozeIds = (_activeSnoozes$map = activeSnoozes === null || activeSnoozes === void 0 ? void 0 : activeSnoozes.map(s => s.id)) !== null && _activeSnoozes$map !== void 0 ? _activeSnoozes$map : [];
  const recurringSnoozesToSkip = [];
  const clearedNonRecurringActiveSnoozes = clearedUnscheduledSnoozes.filter(s => {
    if (!activeSnoozeIds.includes(s.id)) return true;
    // Check if this is a recurring snooze, and return true if so
    if (s.rRule.freq && s.rRule.count !== 1) {
      recurringSnoozesToSkip.push(s.id);
      return true;
    }
  });
  const clearedSnoozesAndSkippedRecurringSnoozes = clearedNonRecurringActiveSnoozes.map(s => {
    var _activeSnoozes$find, _s$skipRecurrences;
    if (s.id && !recurringSnoozesToSkip.includes(s.id)) return s;
    const currentRecurrence = activeSnoozes === null || activeSnoozes === void 0 ? void 0 : (_activeSnoozes$find = activeSnoozes.find(a => a.id === s.id)) === null || _activeSnoozes$find === void 0 ? void 0 : _activeSnoozes$find.lastOccurrence;
    if (!currentRecurrence) return s;
    return {
      ...s,
      skipRecurrences: ((_s$skipRecurrences = s.skipRecurrences) !== null && _s$skipRecurrences !== void 0 ? _s$skipRecurrences : []).concat(currentRecurrence.toISOString())
    };
  });
  return clearedSnoozesAndSkippedRecurringSnoozes;
}
function clearCurrentActiveSnooze(rule) {
  var _activeSnoozes$map2;
  // First attempt to cancel a simple (unscheduled) snooze
  const clearedUnscheduledSnoozes = clearUnscheduledSnooze(rule);
  // Now clear any scheduled snoozes that are currently active and never recur
  const activeSnoozes = (0, _is_rule_snoozed.getActiveScheduledSnoozes)(rule);
  const activeSnoozeIds = (_activeSnoozes$map2 = activeSnoozes === null || activeSnoozes === void 0 ? void 0 : activeSnoozes.map(s => s.id)) !== null && _activeSnoozes$map2 !== void 0 ? _activeSnoozes$map2 : [];
  const recurringSnoozesToSkip = [];
  const clearedNonRecurringActiveSnoozes = clearedUnscheduledSnoozes.filter(s => {
    if (!activeSnoozeIds.includes(s.id)) return true;
    // Check if this is a recurring snooze, and return true if so
    if (s.rRule.freq && s.rRule.count !== 1) {
      recurringSnoozesToSkip.push(s.id);
      return true;
    }
  });
  const clearedSnoozesAndSkippedRecurringSnoozes = clearedNonRecurringActiveSnoozes.map(s => {
    var _activeSnoozes$find2, _s$skipRecurrences2;
    if (s.id && !recurringSnoozesToSkip.includes(s.id)) return s;
    const currentRecurrence = activeSnoozes === null || activeSnoozes === void 0 ? void 0 : (_activeSnoozes$find2 = activeSnoozes.find(a => a.id === s.id)) === null || _activeSnoozes$find2 === void 0 ? void 0 : _activeSnoozes$find2.lastOccurrence;
    if (!currentRecurrence) return s;
    return {
      ...s,
      skipRecurrences: ((_s$skipRecurrences2 = s.skipRecurrences) !== null && _s$skipRecurrences2 !== void 0 ? _s$skipRecurrences2 : []).concat(currentRecurrence.toISOString())
    };
  });
  return clearedSnoozesAndSkippedRecurringSnoozes;
}
function verifySnoozeAttributeScheduleLimit(attributes) {
  var _attributes$snoozeSch;
  const schedules = (_attributes$snoozeSch = attributes.snoozeSchedule) === null || _attributes$snoozeSch === void 0 ? void 0 : _attributes$snoozeSch.filter(snooze => snooze.id);
  if (schedules && schedules.length > 5) {
    throw Error(_i18n.i18n.translate('xpack.alerting.rulesClient.snoozeSchedule.limitReached', {
      defaultMessage: 'Rule cannot have more than 5 snooze schedules'
    }));
  }
}
function verifySnoozeScheduleLimit(snoozeSchedule) {
  const schedules = snoozeSchedule === null || snoozeSchedule === void 0 ? void 0 : snoozeSchedule.filter(snooze => snooze.id);
  if (schedules && schedules.length > 5) {
    throw Error(_i18n.i18n.translate('xpack.alerting.rulesClient.snoozeSchedule.limitReached', {
      defaultMessage: 'Rule cannot have more than 5 snooze schedules'
    }));
  }
}