"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setAlertsToUntracked = setAlertsToUntracked;
var _lodash = require("lodash");
var _ruleDataUtils = require("@kbn/rule-data-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getQuery = ({
  ruleIds = [],
  alertUuids = [],
  alertStatus
}) => {
  const shouldMatchRules = ruleIds.map(ruleId => ({
    term: {
      [_ruleDataUtils.ALERT_RULE_UUID]: {
        value: ruleId
      }
    }
  }));
  const shouldMatchAlerts = alertUuids.map(alertId => ({
    term: {
      [_ruleDataUtils.ALERT_UUID]: {
        value: alertId
      }
    }
  }));
  const statusTerms = [{
    term: {
      [_ruleDataUtils.ALERT_STATUS]: {
        value: alertStatus
      }
    }
  }];
  return [...statusTerms, {
    bool: {
      should: shouldMatchRules
    }
  }, {
    bool: {
      should: shouldMatchAlerts
      // If this is empty, ES will default to minimum_should_match: 0
    }
  }];
};

async function setAlertsToUntracked({
  logger,
  esClient,
  indices,
  ruleIds = [],
  alertUuids = [],
  // OPTIONAL - If no alertUuids are passed, untrack ALL ids by default,
  ensureAuthorized
}) {
  if ((0, _lodash.isEmpty)(ruleIds) && (0, _lodash.isEmpty)(alertUuids)) throw new Error('Must provide either ruleIds or alertUuids');
  if (ensureAuthorized) {
    var _response$aggregation;
    // Fetch all rule type IDs and rule consumers, then run the provided ensureAuthorized check for each of them
    const response = await esClient.search({
      index: indices,
      allow_no_indices: true,
      body: {
        size: 0,
        query: {
          bool: {
            must: getQuery({
              ruleIds,
              alertUuids,
              alertStatus: _ruleDataUtils.ALERT_STATUS_ACTIVE
            })
          }
        },
        aggs: {
          ruleTypeIds: {
            terms: {
              field: _ruleDataUtils.ALERT_RULE_TYPE_ID
            },
            aggs: {
              consumers: {
                terms: {
                  field: _ruleDataUtils.ALERT_RULE_CONSUMER
                }
              }
            }
          }
        }
      }
    });
    const ruleTypeIdBuckets = (_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : _response$aggregation.ruleTypeIds.buckets;
    if (!ruleTypeIdBuckets) throw new Error('Unable to fetch ruleTypeIds for authorization');
    for (const {
      key: ruleTypeId,
      consumers: {
        buckets: consumerBuckets
      }
    } of ruleTypeIdBuckets) {
      const consumers = consumerBuckets.map(b => b.key);
      for (const consumer of consumers) {
        if (consumer === 'siem') throw new Error('Untracking Security alerts is not permitted');
        await ensureAuthorized({
          ruleTypeId,
          consumer
        });
      }
    }
  }
  try {
    // Retry this updateByQuery up to 3 times to make sure the number of documents
    // updated equals the number of documents matched
    let total = 0;
    for (let retryCount = 0; retryCount < 3; retryCount++) {
      var _response$total, _response$updated;
      const response = await esClient.updateByQuery({
        index: indices,
        allow_no_indices: true,
        body: {
          conflicts: 'proceed',
          script: {
            source: getUntrackUpdatePainlessScript(new Date()),
            lang: 'painless'
          },
          query: {
            bool: {
              must: getQuery({
                ruleIds,
                alertUuids,
                alertStatus: _ruleDataUtils.ALERT_STATUS_ACTIVE
              })
            }
          }
        },
        refresh: true
      });
      if (total === 0 && response.total === 0) throw new Error('No active alerts matched the query');
      if (response.total) total = response.total;
      if (response.total === response.updated) break;
      logger.warn(`Attempt ${retryCount + 1}: Failed to untrack ${((_response$total = response.total) !== null && _response$total !== void 0 ? _response$total : 0) - ((_response$updated = response.updated) !== null && _response$updated !== void 0 ? _response$updated : 0)} of ${response.total}; indices ${indices}, ${ruleIds ? 'ruleIds' : 'alertUuids'} ${ruleIds ? ruleIds : alertUuids}`);
    }

    // Fetch and return updated rule and alert instance UUIDs
    const searchResponse = await esClient.search({
      index: indices,
      allow_no_indices: true,
      body: {
        _source: [_ruleDataUtils.ALERT_RULE_UUID, _ruleDataUtils.ALERT_UUID],
        size: total,
        query: {
          bool: {
            must: getQuery({
              ruleIds,
              alertUuids,
              alertStatus: _ruleDataUtils.ALERT_STATUS_UNTRACKED
            })
          }
        }
      }
    });
    return searchResponse.hits.hits.map(hit => hit._source);
  } catch (err) {
    logger.error(`Error marking ${ruleIds ? 'ruleIds' : 'alertUuids'} ${ruleIds ? ruleIds : alertUuids} as untracked - ${err.message}`);
    throw err;
  }
}

// Certain rule types don't flatten their AAD values, apply the ALERT_STATUS key to them directly
const getUntrackUpdatePainlessScript = now => `
if (!ctx._source.containsKey('${_ruleDataUtils.ALERT_STATUS}') || ctx._source['${_ruleDataUtils.ALERT_STATUS}'].empty) {
  ctx._source.${_ruleDataUtils.ALERT_STATUS} = '${_ruleDataUtils.ALERT_STATUS_UNTRACKED}';
  ctx._source.${_ruleDataUtils.ALERT_END} = '${now.toISOString()}';
  ctx._source.${_ruleDataUtils.ALERT_TIME_RANGE}.lte = '${now.toISOString()}';
} else {
  ctx._source['${_ruleDataUtils.ALERT_STATUS}'] = '${_ruleDataUtils.ALERT_STATUS_UNTRACKED}';
  ctx._source['${_ruleDataUtils.ALERT_END}'] = '${now.toISOString()}';
  ctx._source['${_ruleDataUtils.ALERT_TIME_RANGE}'].lte = '${now.toISOString()}';
}`;