"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchIndexInfo = void 0;
var _fieldTypes = require("@kbn/field-types");
var _get_random_docs_request = require("./get_random_docs_request");
var _get_total_doc_count_request = require("./get_total_doc_count_request");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// TODO Consolidate with duplicate `fetchPValues` in
// `x-pack/plugins/apm/server/routes/correlations/queries/fetch_duration_field_candidates.ts`
const SUPPORTED_ES_FIELD_TYPES = [_fieldTypes.ES_FIELD_TYPES.KEYWORD, _fieldTypes.ES_FIELD_TYPES.IP, _fieldTypes.ES_FIELD_TYPES.BOOLEAN];
const SUPPORTED_ES_FIELD_TYPES_TEXT = [_fieldTypes.ES_FIELD_TYPES.TEXT, _fieldTypes.ES_FIELD_TYPES.MATCH_ONLY_TEXT];
const fetchIndexInfo = async (esClient, params, textFieldCandidatesOverrides = [], abortSignal) => {
  const {
    index
  } = params;
  // Get all supported fields
  const respMapping = await esClient.fieldCaps({
    index,
    fields: '*'
  }, {
    signal: abortSignal,
    maxRetries: 0
  });
  const allFieldNames = [];
  const finalFieldCandidates = new Set([]);
  const finalTextFieldCandidates = new Set([]);
  const acceptableFields = new Set();
  const acceptableTextFields = new Set();
  Object.entries(respMapping.fields).forEach(([key, value]) => {
    const fieldTypes = Object.keys(value);
    const isSupportedType = fieldTypes.some(type => SUPPORTED_ES_FIELD_TYPES.includes(type));
    const isAggregatable = fieldTypes.some(type => value[type].aggregatable);
    const isTextField = fieldTypes.some(type => SUPPORTED_ES_FIELD_TYPES_TEXT.includes(type));

    // Check if fieldName is something we can aggregate on
    if (isSupportedType && isAggregatable) {
      acceptableFields.add(key);
    }
    if (isTextField) {
      acceptableTextFields.add(key);
    }
    allFieldNames.push(key);
  });

  // Get the total doc count for the baseline time range
  const respBaselineTotalDocCount = await esClient.search((0, _get_total_doc_count_request.getTotalDocCountRequest)({
    ...params,
    start: params.baselineMin,
    end: params.baselineMax
  }), {
    signal: abortSignal,
    maxRetries: 0
  });

  // Only the deviation window will be used to identify field candidates and sample probability based on total doc count.
  const respDeviationRandomDocs = await esClient.search((0, _get_random_docs_request.getRandomDocsRequest)({
    ...params,
    start: params.deviationMin,
    end: params.deviationMax
  }), {
    signal: abortSignal,
    maxRetries: 0
  });
  const sampledDocs = respDeviationRandomDocs.hits.hits.map(d => {
    var _d$fields;
    return (_d$fields = d.fields) !== null && _d$fields !== void 0 ? _d$fields : {};
  });
  const textFieldCandidatesOverridesWithKeywordPostfix = textFieldCandidatesOverrides.map(d => `${d}.keyword`);

  // Get all field names for each returned doc and flatten it
  // to a list of unique field names used across all docs
  // and filter by list of acceptable fields.
  [...new Set(sampledDocs.map(Object.keys).flat(1))].forEach(field => {
    if (acceptableFields.has(field) && !textFieldCandidatesOverridesWithKeywordPostfix.includes(field)) {
      finalFieldCandidates.add(field);
    }
    if (acceptableTextFields.has(field) && (!allFieldNames.includes(`${field}.keyword`) || textFieldCandidatesOverrides.includes(field))) {
      finalTextFieldCandidates.add(field);
    }
  });
  const baselineTotalDocCount = respBaselineTotalDocCount.hits.total.value;
  const deviationTotalDocCount = respDeviationRandomDocs.hits.total.value;
  return {
    fieldCandidates: [...finalFieldCandidates],
    textFieldCandidates: [...finalTextFieldCandidates],
    baselineTotalDocCount,
    deviationTotalDocCount,
    zeroDocsFallback: baselineTotalDocCount === 0 || deviationTotalDocCount === 0
  };
};
exports.fetchIndexInfo = fetchIndexInfo;