"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCategoryRequest = exports.getBaselineOrDeviationFilter = exports.fetchCategories = void 0;
var _lodash = require("lodash");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _constants = require("../../../../common/constants");
var _create_category_request = require("../../../../common/api/log_categorization/create_category_request");
var _is_request_aborted_error = require("../../../lib/is_request_aborted_error");
var _get_query_with_params = require("./get_query_with_params");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Filter that includes docs from both the baseline and deviation time range.
const getBaselineOrDeviationFilter = params => {
  return {
    bool: {
      should: [{
        range: {
          [params.timeFieldName]: {
            gte: params.baselineMin,
            lte: params.baselineMax,
            format: 'epoch_millis'
          }
        }
      }, {
        range: {
          [params.timeFieldName]: {
            gte: params.deviationMin,
            lte: params.deviationMax,
            format: 'epoch_millis'
          }
        }
      }]
    }
  };
};
exports.getBaselineOrDeviationFilter = getBaselineOrDeviationFilter;
const getCategoryRequest = (params, fieldName, {
  wrap
}) => {
  const {
    index,
    timeFieldName
  } = params;
  const query = (0, _get_query_with_params.getQueryWithParams)({
    params,
    // We're skipping the overall range query here since this
    // is covered by the filter which will match docs in both baseline
    // and deviation time range via `getBaselineOrDeviationFilter`.
    skipRangeQuery: true,
    filter: getBaselineOrDeviationFilter(params)
  });
  const {
    params: request
  } = (0, _create_category_request.createCategoryRequest)(index, fieldName, timeFieldName, undefined, query, wrap);

  // In this case we're only interested in the aggregation which
  // `createCategoryRequest` returns, so we're re-applying the original
  // query we create via `getQueryWithParams` here.
  request.body.query = query;
  return request;
};
exports.getCategoryRequest = getCategoryRequest;
const fetchCategories = async (esClient, params, fieldNames, logger,
// The default value of 1 means no sampling will be used
sampleProbability = 1, emitError, abortSignal) => {
  const randomSamplerWrapper = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
    probability: sampleProbability,
    seed: _constants.RANDOM_SAMPLER_SEED
  });
  const result = [];
  const settledPromises = await Promise.allSettled(fieldNames.map(fieldName => {
    const request = getCategoryRequest(params, fieldName, randomSamplerWrapper);
    return esClient.search(request, {
      signal: abortSignal,
      maxRetries: 0
    });
  }));
  function reportError(fieldName, error) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(error)) {
      logger.error(`Failed to fetch category aggregation for fieldName "${fieldName}", got: \n${JSON.stringify(error, null, 2)}`);
      emitError(`Failed to fetch category aggregation for fieldName "${fieldName}".`);
    }
  }
  for (const [index, settledPromise] of settledPromises.entries()) {
    const fieldName = fieldNames[index];
    if (settledPromise.status === 'rejected') {
      reportError(fieldName, settledPromise.reason);
      // Still continue the analysis even if individual category queries fail.
      continue;
    }
    const resp = settledPromise.value;
    const {
      aggregations
    } = resp;
    if (aggregations === undefined) {
      reportError(fieldName, resp);
      // Still continue the analysis even if individual category queries fail.
      continue;
    }
    const {
      categories: {
        buckets
      }
    } = randomSamplerWrapper.unwrap(aggregations);
    const categories = buckets.map(b => {
      const sparkline = b.sparkline === undefined ? {} : b.sparkline.buckets.reduce((acc2, cur2) => {
        acc2[cur2.key] = cur2.doc_count;
        return acc2;
      }, {});
      return {
        key: b.key,
        count: b.doc_count,
        examples: b.examples.hits.hits.map(h => (0, _lodash.get)(h._source, fieldName)),
        sparkline
      };
    });
    result.push({
      categories
    });
  }
  return result;
};
exports.fetchCategories = fetchCategories;