"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GptEncoding = exports.ALL_SPECIAL_TOKENS = void 0;
/* eslint-disable no-param-reassign */
const BytePairEncodingCore_js_1 = require("./BytePairEncodingCore.js");
const mapping_js_1 = require("./mapping.js");
const modelParams_js_1 = require("./modelParams.js");
const specialTokens_js_1 = require("./specialTokens.js");
const utfUtil_js_1 = require("./utfUtil.js");
const util_js_1 = require("./util.js");
exports.ALL_SPECIAL_TOKENS = 'all';
class GptEncoding {
    static EndOfPrompt = specialTokens_js_1.EndOfPrompt;
    static EndOfText = specialTokens_js_1.EndOfText;
    static FimMiddle = specialTokens_js_1.FimMiddle;
    static FimPrefix = specialTokens_js_1.FimPrefix;
    static FimSuffix = specialTokens_js_1.FimSuffix;
    decoder = new TextDecoder('utf8');
    modelName;
    bytePairEncodingCoreProcessor;
    specialTokenMapping;
    constructor({ tokenSplitRegex, mergeableBytePairRanks, specialTokenMapping, expectedVocabularySize, modelName, }) {
        const maxTokenValue = Math.max((0, util_js_1.getMaxValueFromMap)(mergeableBytePairRanks), (0, util_js_1.getMaxValueFromMap)(specialTokenMapping));
        this.specialTokenMapping = specialTokenMapping;
        if (expectedVocabularySize !== undefined) {
            if (mergeableBytePairRanks.size + specialTokenMapping.size !==
                expectedVocabularySize) {
                throw new Error('The number of mergeable tokens and special tokens must be equal to explicit_n_vocab.');
            }
            if (maxTokenValue !== expectedVocabularySize - 1) {
                throw new Error('The maximum token value must be equal to explicit_n_vocab - 1.');
            }
        }
        this.bytePairEncodingCoreProcessor = new BytePairEncodingCore_js_1.BytePairEncodingCore({
            bytePairEncoder: mergeableBytePairRanks,
            specialTokenEncoder: specialTokenMapping,
            tokenSplitRegex,
        });
        this.encode = this.encode.bind(this);
        this.decode = this.decode.bind(this);
        this.encodeGenerator = this.encodeGenerator.bind(this);
        this.decodeGenerator = this.decodeGenerator.bind(this);
        this.decodeAsyncGenerator = this.decodeAsyncGenerator.bind(this);
        this.isWithinTokenLimit = this.isWithinTokenLimit.bind(this);
        this.encodeChat = this.encodeChat.bind(this);
        this.encodeChatGenerator = this.encodeChatGenerator.bind(this);
        this.modelName = modelName;
    }
    static getEncodingApi(encodingName, getMergeableRanks) {
        const modelParams = (0, modelParams_js_1.getEncodingParams)(encodingName, getMergeableRanks);
        return new GptEncoding(modelParams);
    }
    static getEncodingApiForModel(modelName, getMergeableRanks) {
        const encodingName = mapping_js_1.modelToEncodingMap[modelName];
        const modelParams = (0, modelParams_js_1.getEncodingParams)(encodingName, getMergeableRanks);
        return new GptEncoding({ ...modelParams, modelName });
    }
    static async getEncodingApiAsync(encodingName, getMergeableRanks) {
        const modelParams = await (0, modelParams_js_1.getModelParamsAsync)(encodingName, getMergeableRanks);
        return new GptEncoding(modelParams);
    }
    static async getEncodingApiForModelAsync(modelName, getMergeableRanks) {
        const encodingName = mapping_js_1.modelToEncodingMap[modelName];
        const modelParams = await (0, modelParams_js_1.getModelParamsAsync)(encodingName, getMergeableRanks);
        return new GptEncoding({ ...modelParams, modelName });
    }
    encodeGenerator(lineToEncode, { allowedSpecial = new Set(), disallowedSpecial = new Set([exports.ALL_SPECIAL_TOKENS]), } = {}) {
        const specialTokensSet = new Set(this.specialTokenMapping.keys());
        if (disallowedSpecial.has(exports.ALL_SPECIAL_TOKENS)) {
            disallowedSpecial = new Set(specialTokensSet);
            allowedSpecial.forEach((val) => disallowedSpecial.delete(val));
            disallowedSpecial.forEach((val) => allowedSpecial.delete(val));
        }
        if (allowedSpecial.has(exports.ALL_SPECIAL_TOKENS)) {
            allowedSpecial = specialTokensSet;
        }
        if (disallowedSpecial.size > 0) {
            const regexPattern = (0, util_js_1.getSpecialTokenRegex)(disallowedSpecial);
            const match = lineToEncode.match(regexPattern);
            if (match !== null) {
                throw new Error(`Disallowed special token found: ${match[0]}`);
            }
        }
        return this.bytePairEncodingCoreProcessor.encodeNative(lineToEncode, allowedSpecial);
    }
    encode(lineToEncode, encodeOptions = {}) {
        return [...this.encodeGenerator(lineToEncode, encodeOptions)].flat();
    }
    /**
     * Progressively tokenizes an OpenAI chat.
     * Warning: gpt-3.5-turbo and gpt-4 chat format may change over time.
     * Returns tokens assuming the 'gpt-3.5-turbo-0301' / 'gpt-4-0314' format.
     * Based on OpenAI's guidelines: https://github.com/openai/openai-python/blob/main/chatml.md
     * Also mentioned in section 6 of this document: https://github.com/openai/openai-cookbook/blob/main/examples/How_to_count_tokens_with_tiktoken.ipynb
     */
    *encodeChatGenerator(chat, model = this.modelName) {
        if (!model) {
            throw new Error('Model name must be provided either during initialization or passed in to the method.');
        }
        const params = mapping_js_1.chatModelParams[model];
        const chatStartToken = this.specialTokenMapping.get(specialTokens_js_1.ImStart);
        const chatEndToken = this.specialTokenMapping.get(specialTokens_js_1.ImEnd);
        if (!params || chatStartToken === undefined || chatEndToken === undefined) {
            throw new Error(`Model '${model}' does not support chat.`);
        }
        const allowedSpecial = new Set([specialTokens_js_1.ImSep]);
        const { messageSeparator, roleSeparator } = params;
        const encodedMessageSeparator = messageSeparator.length > 0 ? this.encode(messageSeparator) : [];
        const encodedRoleSeparator = roleSeparator.length > 0
            ? this.encode(roleSeparator, { allowedSpecial })
            : [];
        const nameCache = new Map();
        for (const { role = 'system', name = role, content } of chat) {
            if (content === undefined) {
                throw new Error('Content must be defined for all messages.');
            }
            yield [chatStartToken];
            const encodedName = nameCache.get(name) ?? this.encode(name);
            nameCache.set(name, encodedName);
            yield encodedName;
            if (encodedRoleSeparator.length > 0) {
                yield encodedRoleSeparator;
            }
            yield* this.encodeGenerator(content);
            yield [chatEndToken];
            yield encodedMessageSeparator;
        }
        // every reply is primed with <|start|>assistant<|message|>
        yield [chatStartToken];
        yield* this.encodeGenerator('assistant');
        if (encodedRoleSeparator.length > 0) {
            yield encodedRoleSeparator;
        }
    }
    /**
     * Encodes a chat into a single array of tokens.
     * Warning: gpt-3.5-turbo and gpt-4 chat format may change over time.
     * Returns tokens assuming the 'gpt-3.5-turbo-0301' / 'gpt-4-0314' format.
     * Based on OpenAI's guidelines: https://github.com/openai/openai-python/blob/main/chatml.md
     * Also mentioned in section 6 of this document: https://github.com/openai/openai-cookbook/blob/main/examples/How_to_count_tokens_with_tiktoken.ipynb
     */
    encodeChat(chat, model = this.modelName) {
        return [...this.encodeChatGenerator(chat, model)].flat();
    }
    /**
     * @returns {false | number} false if token limit is exceeded, otherwise the number of tokens
     */
    isWithinTokenLimit(input, tokenLimit) {
        const tokenGenerator = typeof input === 'string'
            ? this.encodeGenerator(input)
            : this.encodeChatGenerator(input);
        let count = 0;
        for (const tokens of tokenGenerator) {
            count += tokens.length;
            if (count > tokenLimit) {
                return false;
            }
        }
        return count;
    }
    *decodeGenerator(inputTokensToDecode) {
        const decodedByteGenerator = this.bytePairEncodingCoreProcessor.decodeNative(inputTokensToDecode);
        let buffer = '';
        for (const decodedPart of decodedByteGenerator) {
            buffer += this.decoder.decode(decodedPart, { stream: true });
            if (buffer.length === 0 || (0, utfUtil_js_1.endsWithIncompleteUtfPairSurrogate)(buffer)) {
                // Keep the high surrogate in the buffer and continue with the next token
                // eslint-disable-next-line no-continue
                continue;
            }
            else {
                yield buffer;
                // reset buffer
                buffer = '';
            }
        }
        // Yield any remaining characters in the buffer
        if (buffer.length > 0) {
            yield buffer;
        }
    }
    async *decodeAsyncGenerator(inputTokensToDecode) {
        const decodedByteGenerator = this.bytePairEncodingCoreProcessor.decodeNativeAsync(inputTokensToDecode);
        let buffer = '';
        for await (const decodedPart of decodedByteGenerator) {
            buffer += this.decoder.decode(decodedPart, { stream: true });
            if (buffer.length === 0 || (0, utfUtil_js_1.endsWithIncompleteUtfPairSurrogate)(buffer)) {
                // Keep the high surrogate in the buffer and continue with the next token
                // eslint-disable-next-line no-continue
                continue;
            }
            else {
                yield buffer;
                // reset buffer
                buffer = '';
            }
        }
        // Yield any remaining characters in the buffer
        if (buffer.length > 0) {
            yield buffer;
        }
    }
    decode(inputTokensToDecode) {
        return [...this.decodeGenerator(inputTokensToDecode)].join('');
    }
}
exports.GptEncoding = GptEncoding;
//# sourceMappingURL=GptEncoding.js.map