/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { __awaiter, __generator, __values } from "tslib";
import { diag, createNoopMeter, } from '@opentelemetry/api';
import { Resource } from '@opentelemetry/resources';
import { MeterProviderSharedState } from './state/MeterProviderSharedState';
import { MetricCollector } from './state/MetricCollector';
/**
 * This class implements the {@link MeterProvider} interface.
 */
var MeterProvider = /** @class */ (function () {
    function MeterProvider(options) {
        var e_1, _a;
        var _b;
        this._shutdown = false;
        var resource = Resource.default().merge((_b = options === null || options === void 0 ? void 0 : options.resource) !== null && _b !== void 0 ? _b : Resource.empty());
        this._sharedState = new MeterProviderSharedState(resource);
        if ((options === null || options === void 0 ? void 0 : options.views) != null && options.views.length > 0) {
            try {
                for (var _c = __values(options.views), _d = _c.next(); !_d.done; _d = _c.next()) {
                    var view = _d.value;
                    this._sharedState.viewRegistry.addView(view);
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
                }
                finally { if (e_1) throw e_1.error; }
            }
        }
    }
    /**
     * Get a meter with the configuration of the MeterProvider.
     */
    MeterProvider.prototype.getMeter = function (name, version, options) {
        if (version === void 0) { version = ''; }
        if (options === void 0) { options = {}; }
        // https://github.com/open-telemetry/opentelemetry-specification/blob/main/specification/metrics/sdk.md#meter-creation
        if (this._shutdown) {
            diag.warn('A shutdown MeterProvider cannot provide a Meter');
            return createNoopMeter();
        }
        return this._sharedState.getMeterSharedState({
            name: name,
            version: version,
            schemaUrl: options.schemaUrl,
        }).meter;
    };
    /**
     * Register a {@link MetricReader} to the meter provider. After the
     * registration, the MetricReader can start metrics collection.
     *
     * @param metricReader the metric reader to be registered.
     */
    MeterProvider.prototype.addMetricReader = function (metricReader) {
        var collector = new MetricCollector(this._sharedState, metricReader);
        metricReader.setMetricProducer(collector);
        this._sharedState.metricCollectors.push(collector);
    };
    /**
     * Flush all buffered data and shut down the MeterProvider and all registered
     * MetricReaders.
     *
     * Returns a promise which is resolved when all flushes are complete.
     */
    MeterProvider.prototype.shutdown = function (options) {
        return __awaiter(this, void 0, void 0, function () {
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (this._shutdown) {
                            diag.warn('shutdown may only be called once per MeterProvider');
                            return [2 /*return*/];
                        }
                        this._shutdown = true;
                        return [4 /*yield*/, Promise.all(this._sharedState.metricCollectors.map(function (collector) {
                                return collector.shutdown(options);
                            }))];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Notifies all registered MetricReaders to flush any buffered data.
     *
     * Returns a promise which is resolved when all flushes are complete.
     */
    MeterProvider.prototype.forceFlush = function (options) {
        return __awaiter(this, void 0, void 0, function () {
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        // do not flush after shutdown
                        if (this._shutdown) {
                            diag.warn('invalid attempt to force flush after MeterProvider shutdown');
                            return [2 /*return*/];
                        }
                        return [4 /*yield*/, Promise.all(this._sharedState.metricCollectors.map(function (collector) {
                                return collector.forceFlush(options);
                            }))];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    return MeterProvider;
}());
export { MeterProvider };
//# sourceMappingURL=MeterProvider.js.map