"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useColumns = void 0;
var _react = require("react");
var _lodash = require("lodash");
var _toggle_column = require("./toggle_column");
var _use_fetch_browser_fields_capabilities = require("../use_fetch_browser_fields_capabilities");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fieldTypeToDataGridColumnTypeMapper = fieldType => {
  if (fieldType === 'date') return 'datetime';
  if (fieldType === 'number') return 'numeric';
  if (fieldType === 'object') return 'json';
  return fieldType;
};
const getFieldCategoryFromColumnId = columnId => {
  const fieldName = columnId.split('.');
  if (fieldName.length === 1) {
    return 'base';
  }
  return fieldName[0];
};

/**
 * EUI Data Grid expects the columns to have a property 'schema' defined for proper sorting
 * this schema as its own types as can be check out in the docs so we add it here manually
 * https://eui.elastic.co/#/tabular-content/data-grid-schema-columns
 */
const euiColumnFactory = (columnId, browserFields, defaultColumns) => {
  const defaultColumn = getColumnByColumnId(defaultColumns, columnId);
  const column = defaultColumn ? defaultColumn : {
    id: columnId
  };
  const browserFieldsProps = getBrowserFieldProps(columnId, browserFields);
  return {
    ...column,
    schema: fieldTypeToDataGridColumnTypeMapper(browserFieldsProps.type)
  };
};
const getBrowserFieldProps = (columnId, browserFields) => {
  const notFoundSpecs = {
    type: 'string'
  };
  if (!browserFields || Object.keys(browserFields).length === 0) {
    return notFoundSpecs;
  }
  const category = getFieldCategoryFromColumnId(columnId);
  if (!browserFields[category]) {
    return notFoundSpecs;
  }
  const categorySpecs = browserFields[category].fields;
  if (!categorySpecs) {
    return notFoundSpecs;
  }
  const fieldSpecs = categorySpecs[columnId];
  return fieldSpecs ? fieldSpecs : notFoundSpecs;
};
const isPopulatedColumn = column => Boolean(column.schema);

/**
 * @param columns Columns to be considered in the alerts table
 * @param browserFields constant object with all field capabilities
 * @returns columns but with the info needed by the data grid to work as expected, e.g sorting
 */
const populateColumns = (columns, browserFields, defaultColumns) => {
  return columns.map(column => {
    return isPopulatedColumn(column) ? column : euiColumnFactory(column.id, browserFields, defaultColumns);
  });
};
const getColumnIds = columns => {
  return columns.map(column => column.id);
};
const getColumnByColumnId = (columns, columnId) => {
  return columns.find(({
    id
  }) => id === columnId);
};
const persist = ({
  id,
  storageAlertsTable,
  columns,
  storage,
  visibleColumns
}) => {
  storageAlertsTable.current = {
    ...storageAlertsTable.current,
    columns,
    visibleColumns
  };
  storage.current.set(id, storageAlertsTable.current);
};
const useColumns = ({
  featureIds,
  storageAlertsTable,
  storage,
  id,
  defaultColumns,
  initialBrowserFields
}) => {
  var _storageAlertsTable$c;
  const [isBrowserFieldDataLoading, browserFields] = (0, _use_fetch_browser_fields_capabilities.useFetchBrowserFieldCapabilities)({
    featureIds,
    initialBrowserFields
  });
  const [columns, setColumns] = (0, _react.useState)(() => {
    let cols = storageAlertsTable.current.columns;
    // before restoring from storage, enrich the column data
    if (initialBrowserFields && defaultColumns) {
      cols = populateColumns(cols, initialBrowserFields, defaultColumns);
    } else if (cols && cols.length === 0) {
      cols = defaultColumns;
    }
    return cols;
  });
  const [visibleColumns, setVisibleColumns] = (0, _react.useState)((_storageAlertsTable$c = storageAlertsTable.current.visibleColumns) !== null && _storageAlertsTable$c !== void 0 ? _storageAlertsTable$c : getColumnIds(columns));
  const [isColumnsPopulated, setColumnsPopulated] = (0, _react.useState)(false);
  const defaultColumnsRef = (0, _react.useRef)(defaultColumns);
  const didDefaultColumnChange = defaultColumns !== defaultColumnsRef.current;
  const setColumnsByColumnIds = (0, _react.useCallback)(columnIds => {
    setVisibleColumns(columnIds);
    persist({
      id,
      storage,
      storageAlertsTable,
      columns,
      visibleColumns: columnIds
    });
  }, [columns, id, storage, storageAlertsTable]);
  (0, _react.useEffect)(() => {
    // if defaultColumns have changed,
    // get the latest columns provided by client and
    if (didDefaultColumnChange && defaultColumnsRef.current) {
      var _storageAlertsTable$c2;
      defaultColumnsRef.current = defaultColumns;
      setColumnsPopulated(false);
      // storageAlertTable already account for the changes in defaultColumns
      // Technically storageAlertsTable = localStorageData ?? defaultColumns
      setColumns(storageAlertsTable.current.columns);
      setVisibleColumns((_storageAlertsTable$c2 = storageAlertsTable.current.visibleColumns) !== null && _storageAlertsTable$c2 !== void 0 ? _storageAlertsTable$c2 : visibleColumns);
      return;
    }
  }, [didDefaultColumnChange, storageAlertsTable, defaultColumns, visibleColumns]);
  (0, _react.useEffect)(() => {
    if ((0, _lodash.isEmpty)(browserFields) || isColumnsPopulated) return;
    const populatedColumns = populateColumns(columns, browserFields, defaultColumns);
    setColumnsPopulated(true);
    setColumns(populatedColumns);
  }, [browserFields, defaultColumns, isBrowserFieldDataLoading, isColumnsPopulated, columns]);
  const setColumnsAndSave = (0, _react.useCallback)((newColumns, newVisibleColumns) => {
    setColumns(newColumns);
    persist({
      id,
      storage,
      storageAlertsTable,
      columns: newColumns,
      visibleColumns: newVisibleColumns
    });
  }, [id, storage, storageAlertsTable]);
  const onToggleColumn = (0, _react.useCallback)(columnId => {
    const column = euiColumnFactory(columnId, browserFields, defaultColumns);
    const newColumns = (0, _toggle_column.toggleColumn)({
      column,
      columns,
      defaultColumns
    });
    let newVisibleColumns = visibleColumns;
    if (visibleColumns.includes(columnId)) {
      newVisibleColumns = visibleColumns.filter(vc => vc !== columnId);
    } else {
      newVisibleColumns = [visibleColumns[0], columnId, ...visibleColumns.slice(1)];
    }
    setVisibleColumns(newVisibleColumns);
    setColumnsAndSave(newColumns, newVisibleColumns);
  }, [browserFields, columns, defaultColumns, setColumnsAndSave, visibleColumns]);
  const onResetColumns = (0, _react.useCallback)(() => {
    const populatedDefaultColumns = populateColumns(defaultColumns, browserFields, defaultColumns);
    setColumnsAndSave(populatedDefaultColumns, populatedDefaultColumns.map(pdc => pdc.id));
  }, [browserFields, defaultColumns, setColumnsAndSave]);
  const onColumnResize = (0, _react.useCallback)(({
    columnId,
    width
  }) => {
    const colIndex = columns.findIndex(c => c.id === columnId);
    if (colIndex > -1) {
      columns.splice(colIndex, 1, {
        ...columns[colIndex],
        initialWidth: width
      });
      setColumnsAndSave(columns, visibleColumns);
    }
  }, [columns, setColumnsAndSave, visibleColumns]);
  const fieldsToFetch = (0, _react.useMemo)(() => [...columns.map(col => ({
    field: col.id,
    include_unmapped: true
  }))], [columns]);
  return (0, _react.useMemo)(() => ({
    columns,
    visibleColumns,
    isBrowserFieldDataLoading,
    browserFields,
    onToggleColumn,
    onResetColumns,
    onChangeVisibleColumns: setColumnsByColumnIds,
    onColumnResize,
    fields: fieldsToFetch
  }), [browserFields, columns, fieldsToFetch, isBrowserFieldDataLoading, onColumnResize, onResetColumns, onToggleColumn, setColumnsByColumnIds, visibleColumns]);
};
exports.useColumns = useColumns;