"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MAX_RETRIES = void 0;
exports.retryableBulkUpdate = retryableBulkUpdate;
var _result_type = require("./result_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_RETRIES = 2;
exports.MAX_RETRIES = MAX_RETRIES;
async function retryableBulkUpdate({
  taskIds,
  getTasks,
  filter,
  map,
  store,
  validate
}) {
  const resultMap = {};
  async function attemptToUpdate(taskIdsToAttempt) {
    const tasksToUpdate = (await getTasks(taskIdsToAttempt)).reduce((acc, task) => {
      if ((0, _result_type.isErr)(task)) {
        resultMap[task.error.id] = buildBulkUpdateErr(task.error);
      } else {
        acc.push(task.value);
      }
      return acc;
    }, []).filter(filter).map(map);
    const bulkUpdateResult = await store.bulkUpdate(tasksToUpdate, {
      validate
    });
    for (const result of bulkUpdateResult) {
      const taskId = getId(result);
      resultMap[taskId] = result;
    }
  }
  await attemptToUpdate(taskIds);
  let retry = 1;
  while (retry++ <= MAX_RETRIES && getRetriableTaskIds(resultMap).length > 0) {
    await attemptToUpdate(getRetriableTaskIds(resultMap));
  }
  return Object.values(resultMap).reduce((acc, result) => {
    if ((0, _result_type.isOk)(result)) {
      acc.tasks.push(result.value);
    } else {
      acc.errors.push(result.error);
    }
    return acc;
  }, {
    tasks: [],
    errors: []
  });
}
function getId(bulkUpdateResult) {
  return (0, _result_type.isOk)(bulkUpdateResult) ? bulkUpdateResult.value.id : bulkUpdateResult.error.id;
}
function getRetriableTaskIds(resultMap) {
  return Object.values(resultMap).filter(result => (0, _result_type.isErr)(result) && result.error.error.statusCode === 409).map(result => getId(result));
}
function buildBulkUpdateErr(error) {
  return (0, _result_type.asErr)({
    id: error.id,
    type: error.type,
    error: {
      error: error.error.error,
      statusCode: error.error.statusCode,
      message: error.error.message,
      ...(error.error.metadata ? error.error.metadata : {})
    }
  });
}