"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPrivateLocationsRoute = exports.getPrivateLocationsAndAgentPolicies = void 0;
var _server = require("@kbn/core/server");
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../../common/constants");
var _get_private_locations = require("../../../synthetics_service/get_private_locations");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getPrivateLocationsRoute = () => ({
  method: 'GET',
  path: _constants.SYNTHETICS_API_URLS.PRIVATE_LOCATIONS + '/{id?}',
  validate: {},
  validation: {
    request: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  },
  handler: async ({
    savedObjectsClient,
    syntheticsMonitorClient,
    request,
    response
  }) => {
    const {
      id
    } = request.params;
    const {
      locations,
      agentPolicies
    } = await getPrivateLocationsAndAgentPolicies(savedObjectsClient, syntheticsMonitorClient);
    const list = (0, _helpers.toClientContract)({
      locations
    }, agentPolicies);
    if (!id) return list;
    const location = list.find(loc => loc.id === id || loc.label === id);
    if (!location) {
      return response.notFound({
        body: {
          message: `Private location with id or label "${id}" not found`
        }
      });
    }
    return location;
  }
});
exports.getPrivateLocationsRoute = getPrivateLocationsRoute;
const getPrivateLocationsAndAgentPolicies = async (savedObjectsClient, syntheticsMonitorClient) => {
  try {
    const [privateLocations, agentPolicies] = await Promise.all([(0, _get_private_locations.getPrivateLocations)(savedObjectsClient), syntheticsMonitorClient.privateLocationAPI.getAgentPolicies()]);
    return {
      locations: privateLocations || [],
      agentPolicies
    };
  } catch (getErr) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(getErr)) {
      return {
        locations: [],
        agentPolicies: []
      };
    }
    throw getErr;
  }
};
exports.getPrivateLocationsAndAgentPolicies = getPrivateLocationsAndAgentPolicies;