"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createGetCurrentStatusRoute = void 0;
exports.getStatus = getStatus;
exports.periodToMs = periodToMs;
var _lodash = require("lodash");
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _moment = _interopRequireDefault(require("moment"));
var _runtime_types = require("../../../common/runtime_types");
var _get_all_monitors = require("../../saved_objects/synthetics_monitor/get_all_monitors");
var _query_monitor_status = require("../../queries/query_monitor_status");
var _constants = require("../../../common/constants");
var _common = require("../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Helper function that converts a monitor's schedule to a value to use to generate
 * an appropriate look-back window for snapshot count.
 * @param schedule a number/unit pair that represents how often a configured monitor runs
 * @returns schedule interval in ms
 */
function periodToMs(schedule) {
  if (Object.keys(_datemath.default.unitsMap).indexOf(schedule.unit) === -1) return 0;
  return parseInt(schedule.number, 10) * _datemath.default.unitsMap[schedule.unit].base;
}

/**
 * Multi-stage function that first queries all the user's saved object monitor configs.
 *
 * Subsequently, fetch the status for each monitor per location in the data streams.
 * @returns The counts of up/down/disabled monitor by location, and a map of each monitor:location status.
 */
async function getStatus(context, params) {
  const {
    uptimeEsClient,
    syntheticsMonitorClient,
    savedObjectsClient,
    server
  } = context;
  const {
    query,
    scopeStatusByLocation = true
  } = params;

  /**
   * Walk through all monitor saved objects, bucket IDs by disabled/enabled status.
   *
   * Track max period to make sure the snapshot query should reach back far enough to catch
   * latest ping for all enabled monitors.
   */

  const {
    filtersStr,
    locationFilter: queryLocations
  } = await (0, _common.getMonitorFilters)({
    ...params,
    context
  });
  const allMonitors = await (0, _get_all_monitors.getAllMonitors)({
    soClient: savedObjectsClient,
    search: query ? `${query}*` : undefined,
    filter: filtersStr,
    fields: [_runtime_types.ConfigKey.ENABLED, _runtime_types.ConfigKey.LOCATIONS, _runtime_types.ConfigKey.MONITOR_QUERY_ID, _runtime_types.ConfigKey.CONFIG_ID, _runtime_types.ConfigKey.SCHEDULE, _runtime_types.ConfigKey.MONITOR_SOURCE_TYPE]
  });
  const {
    enabledMonitorQueryIds,
    disabledMonitorQueryIds,
    allIds,
    disabledCount,
    maxPeriod,
    monitorLocationIds,
    monitorLocationMap,
    disabledMonitorsCount,
    projectMonitorsCount,
    monitorQueryIdToConfigIdMap
  } = (0, _get_all_monitors.processMonitors)(allMonitors, server, savedObjectsClient, syntheticsMonitorClient, queryLocations);

  // Account for locations filter
  const listOfLocationAfterFilter = queryLocations && scopeStatusByLocation ? (0, _lodash.intersection)(monitorLocationIds, queryLocations) : monitorLocationIds;
  const range = {
    from: (0, _moment.default)().subtract(maxPeriod, 'milliseconds').subtract(20, 'minutes').toISOString(),
    to: 'now'
  };
  const {
    up,
    down,
    pending,
    upConfigs,
    downConfigs,
    pendingConfigs
  } = await (0, _query_monitor_status.queryMonitorStatus)(uptimeEsClient, listOfLocationAfterFilter, range, enabledMonitorQueryIds, monitorLocationMap, monitorQueryIdToConfigIdMap);
  return {
    allIds,
    allMonitorsCount: allMonitors.length,
    disabledMonitorsCount,
    projectMonitorsCount,
    enabledMonitorQueryIds,
    disabledMonitorQueryIds,
    disabledCount,
    up,
    down,
    pending,
    upConfigs,
    downConfigs,
    pendingConfigs
  };
}
const createGetCurrentStatusRoute = () => ({
  method: 'GET',
  path: _constants.SYNTHETICS_API_URLS.OVERVIEW_STATUS,
  validate: {
    query: _common.OverviewStatusSchema
  },
  handler: async routeContext => {
    const {
      request
    } = routeContext;
    const params = request.query;
    return await getStatus(routeContext, params);
  }
});
exports.createGetCurrentStatusRoute = createGetCurrentStatusRoute;