"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.shouldCompose = shouldCompose;
exports.useComposeImageFromRef = void 0;
var _react = require("react");
var _reactRedux = require("react-redux");
var _compose_screenshot_images = require("../utils/monitor_test_result/compose_screenshot_images");
var _state = require("../state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function allBlocksLoaded(blocks, hashes) {
  for (const hash of hashes) {
    if (!blocks[hash] || (0, _state.isPendingBlock)(blocks[hash])) {
      return false;
    }
  }
  return true;
}

/**
 * Checks if two refs are the same. If the ref is unchanged, there's no need
 * to run the expensive draw procedure.
 *
 * The key fields here are `step.index` and `check_group`, as there's a 1:1 between
 * journey and check group, and each step has a unique index within a journey.
 */
const isNewRef = ({
  ref: {
    screenshotRef: {
      synthetics: {
        step: {
          index: indexA
        }
      },
      monitor: {
        check_group: checkGroupA
      }
    }
  }
}, {
  ref: {
    screenshotRef: {
      synthetics: {
        step: {
          index: indexB
        }
      },
      monitor: {
        check_group: checkGroupB
      }
    }
  }
}) => indexA !== indexB || checkGroupA !== checkGroupB;
function shouldCompose(imageData, imgRef, curRef, blocks) {
  return allBlocksLoaded(blocks, imgRef.ref.screenshotRef.screenshot_ref.blocks.map(({
    hash
  }) => hash)) && (typeof imageData === 'undefined' || isNewRef(imgRef, curRef));
}

/**
 * Assembles the data for a composite image and returns the image src and isComposing.
 * @param imgRef the data and dimensions for the composite image.
 */
const useComposeImageFromRef = imgRef => {
  var _ref, _imgRef$stepName, _imgRef$ref$screensho, _composeState$uniqueR;
  const dispatch = (0, _reactRedux.useDispatch)();
  const {
    blocks
  } = (0, _reactRedux.useSelector)(_state.selectBrowserJourneyState);
  const [isAnyBlockLoading, isAllBlocksLoaded] = (0, _react.useMemo)(() => [getIsAnyBlockLoading(imgRef, blocks), getIsAllBlocksLoaded(imgRef, blocks)], [imgRef, blocks]);
  (0, _react.useEffect)(() => {
    if (imgRef) {
      dispatch((0, _state.fetchBlocksAction)(imgRef.ref.screenshotRef.screenshot_ref.blocks.map(({
        hash
      }) => hash)));
    }
  }, [dispatch, imgRef]);
  const stepRefId = (_ref = (_imgRef$stepName = imgRef === null || imgRef === void 0 ? void 0 : imgRef.stepName) !== null && _imgRef$stepName !== void 0 ? _imgRef$stepName : imgRef === null || imgRef === void 0 ? void 0 : (_imgRef$ref$screensho = imgRef.ref.screenshotRef.screenshot_ref.blocks[0]) === null || _imgRef$ref$screensho === void 0 ? void 0 : _imgRef$ref$screensho.hash) !== null && _ref !== void 0 ? _ref : '';
  const uniqueRefId = `${imgRef === null || imgRef === void 0 ? void 0 : imgRef.ref.screenshotRef.monitor.check_group}/${stepRefId}`;
  const [composeState, setComposeState] = (0, _react.useState)({});
  const {
    isComposing,
    imgSrc
  } = (_composeState$uniqueR = composeState[uniqueRefId]) !== null && _composeState$uniqueR !== void 0 ? _composeState$uniqueR : {
    isComposing: false,
    imgSrc: undefined
  };
  (0, _react.useEffect)(() => {
    if (imgRef) {
      // if the imgSrc is truthy it means it's already been composed, so there
      // is no need to call the function
      if (!isComposing && !imgSrc && isAllBlocksLoaded) {
        setComposeState(prevState => ({
          ...prevState,
          [uniqueRefId]: {
            isComposing: true,
            imgSrc: undefined
          }
        }));
        const canvas = document.createElement('canvas');
        (0, _compose_screenshot_images.composeScreenshotRef)(imgRef, canvas, blocks).then(() => {
          var _canvas$parentElement;
          const dataUrl = canvas.toDataURL('image/jpg', 1.0);
          setComposeState(prevState => ({
            ...prevState,
            [uniqueRefId]: {
              isComposing: false,
              imgSrc: dataUrl
            }
          }));
          (_canvas$parentElement = canvas.parentElement) === null || _canvas$parentElement === void 0 ? void 0 : _canvas$parentElement.removeChild(canvas);
        });
      }
    }
  }, [blocks, imgRef, composeState, imgSrc, isComposing, uniqueRefId, isAnyBlockLoading, isAllBlocksLoaded]);
  return {
    isComposing: isComposing || isAnyBlockLoading,
    imgSrc
  };
};
exports.useComposeImageFromRef = useComposeImageFromRef;
function getIsAnyBlockLoading(imgRef, blocks) {
  if (!imgRef) {
    return false;
  }
  const hashes = imgRef.ref.screenshotRef.screenshot_ref.blocks.map(({
    hash
  }) => hash);
  for (const hash of hashes) {
    if (blocks[hash] && (0, _state.isPendingBlock)(blocks[hash])) {
      return true;
    }
  }
  return false;
}
function getIsAllBlocksLoaded(imgRef, blocks) {
  if (!imgRef) {
    return false;
  }
  const hashes = imgRef.ref.screenshotRef.screenshot_ref.blocks.map(({
    hash
  }) => hash);
  for (const hash of hashes) {
    if (!blocks[hash] || (0, _state.isPendingBlock)(blocks[hash])) {
      return false;
    }
  }
  return true;
}