"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startTransform = exports.scheduleTransformNow = exports.scheduleLatestTransformNow = exports.removeLegacyTransforms = exports.getLegacyTransforms = exports.getLatestTransformId = exports.createTransform = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _search_strategy = require("../../../../common/search_strategy");
var _risk_score_modules = require("../../../../common/utils/risk_score_modules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getLegacyTransforms = async ({
  namespace,
  esClient
}) => {
  const getTransformStatsRequests = [];
  [_search_strategy.RiskScoreEntity.host, _search_strategy.RiskScoreEntity.user].forEach(entity => {
    getTransformStatsRequests.push(esClient.transform.getTransform({
      transform_id: (0, _risk_score_modules.getRiskScorePivotTransformId)(entity, namespace)
    }));
    getTransformStatsRequests.push(esClient.transform.getTransform({
      transform_id: (0, _risk_score_modules.getRiskScoreLatestTransformId)(entity, namespace)
    }));
  });
  const results = await Promise.allSettled(getTransformStatsRequests);
  const transforms = results.reduce((acc, result) => {
    var _result$value, _result$value$transfo;
    if (result.status === 'fulfilled' && ((_result$value = result.value) === null || _result$value === void 0 ? void 0 : (_result$value$transfo = _result$value.transforms) === null || _result$value$transfo === void 0 ? void 0 : _result$value$transfo.length) > 0) {
      acc.push(...result.value.transforms);
    }
    return acc;
  }, []);
  return transforms;
};
exports.getLegacyTransforms = getLegacyTransforms;
const removeLegacyTransforms = async ({
  namespace,
  esClient
}) => {
  const transforms = await getLegacyTransforms({
    namespace,
    esClient
  });
  const stopTransformRequests = transforms.map(t => esClient.transform.deleteTransform({
    transform_id: t.id,
    force: true
  }));
  await Promise.allSettled(stopTransformRequests);
};
exports.removeLegacyTransforms = removeLegacyTransforms;
const createTransform = async ({
  esClient,
  transform,
  logger
}) => {
  try {
    await esClient.transform.getTransform({
      transform_id: transform.transform_id
    });
    logger.info(`Transform ${transform.transform_id} already exists`);
  } catch (existErr) {
    const transformedError = (0, _securitysolutionEsUtils.transformError)(existErr);
    if (transformedError.statusCode === 404) {
      return esClient.transform.putTransform(transform);
    } else {
      logger.error(`Failed to check if transform ${transform.transform_id} exists before creation: ${transformedError.message}`);
      throw existErr;
    }
  }
};
exports.createTransform = createTransform;
const getLatestTransformId = namespace => `risk_score_latest_transform_${namespace}`;
exports.getLatestTransformId = getLatestTransformId;
const hasTransformStarted = transformStats => {
  return transformStats.state === 'indexing' || transformStats.state === 'started';
};
const startTransform = async ({
  esClient,
  transformId
}) => {
  const transformStats = await esClient.transform.getTransformStats({
    transform_id: transformId
  });
  if (transformStats.count <= 0) {
    throw new Error(`Unable to find transform status for [${transformId}] while attempting to start`);
  }
  if (hasTransformStarted(transformStats.transforms[0])) {
    return;
  }
  return esClient.transform.startTransform({
    transform_id: transformId
  });
};
exports.startTransform = startTransform;
const scheduleTransformNow = async ({
  esClient,
  transformId
}) => {
  const transformStats = await esClient.transform.getTransformStats({
    transform_id: transformId
  });
  if (transformStats.count <= 0) {
    throw new Error(`Unable to find transform status for [${transformId}] while attempting to schedule now`);
  }
  if (hasTransformStarted(transformStats.transforms[0])) {
    await esClient.transform.scheduleNowTransform({
      transform_id: transformId
    });
  } else {
    await esClient.transform.startTransform({
      transform_id: transformId
    });
  }
};
exports.scheduleTransformNow = scheduleTransformNow;
const scheduleLatestTransformNow = async ({
  namespace,
  esClient
}) => {
  const transformId = getLatestTransformId(namespace);
  await scheduleTransformNow({
    esClient,
    transformId
  });
};
exports.scheduleLatestTransformNow = scheduleLatestTransformNow;