"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startRiskScoringTask = exports.runTask = exports.removeRiskScoringTask = exports.registerRiskScoringTask = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _std = require("@kbn/std");
var _server = require("@kbn/core/server");
var _risk_score_service = require("../risk_score_service");
var _risk_engine_data_client = require("../../risk_engine/risk_engine_data_client");
var _risk_score_data_client = require("../risk_score_data_client");
var _helpers = require("../helpers");
var _state = require("./state");
var _constants = require("./constants");
var _helpers2 = require("./helpers");
var _types = require("../../../../../common/risk_engine/types");
var _events = require("../../../telemetry/event_based/events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const logFactory = (logger, taskId) => message => logger.info(`[task ${taskId}]: ${message}`);
const getTaskName = () => _constants.TYPE;
const getTaskId = namespace => `${_constants.TYPE}:${namespace}:${_constants.VERSION}`;
const registerRiskScoringTask = ({
  getStartServices,
  kibanaVersion,
  logger,
  taskManager,
  telemetry
}) => {
  if (!taskManager) {
    logger.info('Task Manager is unavailable; skipping risk engine task registration.');
    return;
  }
  const getRiskScoreService = namespace => getStartServices().then(([coreStart, _]) => {
    const esClient = coreStart.elasticsearch.client.asInternalUser;
    const soClient = (0, _helpers2.buildScopedInternalSavedObjectsClientUnsafe)({
      coreStart,
      namespace
    });
    const riskEngineDataClient = new _risk_engine_data_client.RiskEngineDataClient({
      logger,
      kibanaVersion,
      esClient,
      namespace,
      soClient
    });
    const riskScoreDataClient = new _risk_score_data_client.RiskScoreDataClient({
      logger,
      kibanaVersion,
      esClient,
      namespace,
      soClient
    });
    return (0, _risk_score_service.riskScoreServiceFactory)({
      esClient,
      logger,
      riskEngineDataClient,
      riskScoreDataClient,
      spaceId: namespace
    });
  });
  taskManager.registerTaskDefinitions({
    [getTaskName()]: {
      title: 'Entity Analytics Risk Engine - Risk Scoring Task',
      timeout: _constants.TIMEOUT,
      stateSchemaByVersion: _state.stateSchemaByVersion,
      createTaskRunner: createTaskRunnerFactory({
        logger,
        getRiskScoreService,
        telemetry
      })
    }
  });
};
exports.registerRiskScoringTask = registerRiskScoringTask;
const startRiskScoringTask = async ({
  logger,
  namespace,
  riskEngineDataClient,
  taskManager
}) => {
  var _await$riskEngineData, _await$riskEngineData2;
  const taskId = getTaskId(namespace);
  const log = logFactory(logger, taskId);
  log('starting task');
  const interval = (_await$riskEngineData = (_await$riskEngineData2 = await riskEngineDataClient.getConfiguration()) === null || _await$riskEngineData2 === void 0 ? void 0 : _await$riskEngineData2.interval) !== null && _await$riskEngineData !== void 0 ? _await$riskEngineData : _constants.INTERVAL;
  log('attempting to schedule');
  try {
    await taskManager.ensureScheduled({
      id: taskId,
      taskType: getTaskName(),
      scope: _constants.SCOPE,
      schedule: {
        interval
      },
      state: {
        ..._state.defaultState,
        namespace
      },
      params: {
        version: _constants.VERSION
      }
    });
  } catch (e) {
    logger.warn(`[task ${taskId}]: error scheduling task, received ${e.message}`);
    throw e;
  }
};
exports.startRiskScoringTask = startRiskScoringTask;
const removeRiskScoringTask = async ({
  logger,
  namespace,
  taskManager
}) => {
  try {
    await taskManager.remove(getTaskId(namespace));
  } catch (err) {
    if (!_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      logger.error(`Failed to remove risk scoring task: ${err.message}`);
      throw err;
    }
  }
};
exports.removeRiskScoringTask = removeRiskScoringTask;
const runTask = async ({
  getRiskScoreService,
  isCancelled,
  logger,
  taskInstance,
  telemetry
}) => {
  const state = taskInstance.state;
  const taskId = taskInstance.id;
  const log = logFactory(logger, taskId);
  try {
    var _taskInstance$schedul;
    const taskStartTime = (0, _moment.default)().utc().toISOString();
    log('running task');
    let scoresWritten = 0;
    const updatedState = {
      lastExecutionTimestamp: taskStartTime,
      namespace: state.namespace,
      runs: state.runs + 1,
      scoresWritten
    };
    if (taskId !== getTaskId(state.namespace)) {
      log('outdated task; exiting');
      return {
        state: updatedState
      };
    }
    const riskScoreService = await getRiskScoreService(state.namespace);
    if (!riskScoreService) {
      log('risk score service is not available; exiting task');
      return {
        state: updatedState
      };
    }
    const configuration = await riskScoreService.getConfiguration();
    if (configuration == null) {
      log('Risk engine configuration not found; exiting task. Please reinitialize the risk engine and try again');
      return {
        state: updatedState
      };
    }
    const {
      dataViewId,
      enabled,
      filter,
      identifierType: configuredIdentifierType,
      range: configuredRange,
      pageSize
    } = configuration;
    if (!enabled) {
      log('risk engine is not enabled, exiting task');
      return {
        state: updatedState
      };
    }
    const range = (0, _helpers2.convertRangeToISO)(configuredRange);
    const {
      index,
      runtimeMappings
    } = await riskScoreService.getRiskInputsIndex({
      dataViewId
    });
    const identifierTypes = configuredIdentifierType ? [configuredIdentifierType] : [_types.RiskScoreEntity.host, _types.RiskScoreEntity.user];
    await (0, _std.asyncForEach)(identifierTypes, async identifierType => {
      let isWorkComplete = isCancelled();
      let afterKeys = {};
      while (!isWorkComplete) {
        const result = await riskScoreService.calculateAndPersistScores({
          afterKeys,
          index,
          filter,
          identifierType,
          pageSize,
          range,
          runtimeMappings,
          weights: []
        });
        isWorkComplete = (0, _helpers.isRiskScoreCalculationComplete)(result) || isCancelled();
        afterKeys = result.after_keys;
        scoresWritten += result.scores_written;
      }
    });
    updatedState.scoresWritten = scoresWritten;
    const taskCompletionTime = (0, _moment.default)().utc().toISOString();
    const taskDurationInSeconds = (0, _moment.default)(taskCompletionTime).diff((0, _moment.default)(taskStartTime), 'seconds');
    const telemetryEvent = {
      scoresWritten,
      taskDurationInSeconds,
      interval: taskInstance === null || taskInstance === void 0 ? void 0 : (_taskInstance$schedul = taskInstance.schedule) === null || _taskInstance$schedul === void 0 ? void 0 : _taskInstance$schedul.interval
    };
    telemetry.reportEvent(_events.RISK_SCORE_EXECUTION_SUCCESS_EVENT.eventType, telemetryEvent);
    riskScoreService.scheduleLatestTransformNow();
    if (isCancelled()) {
      log('task was cancelled');
      telemetry.reportEvent(_events.RISK_SCORE_EXECUTION_CANCELLATION_EVENT.eventType, telemetryEvent);
    }
    log('task run completed');
    log(JSON.stringify(telemetryEvent));
    return {
      state: updatedState
    };
  } catch (e) {
    telemetry.reportEvent(_events.RISK_SCORE_EXECUTION_ERROR_EVENT.eventType, {});
    throw e;
  }
};
exports.runTask = runTask;
const createTaskRunnerFactory = ({
  logger,
  getRiskScoreService,
  telemetry
}) => ({
  taskInstance
}) => {
  let cancelled = false;
  const isCancelled = () => cancelled;
  return {
    run: async () => runTask({
      getRiskScoreService,
      isCancelled,
      logger,
      taskInstance,
      telemetry
    }),
    cancel: async () => {
      cancelled = true;
    }
  };
};