"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.searchAfterAndBulkCreate = void 0;
var _lodash = require("lodash");
var _single_search_after = require("./single_search_after");
var _filter_events_against_list = require("./large_list_filters/filter_events_against_list");
var _send_telemetry_events = require("./send_telemetry_events");
var _utils = require("./utils");
var _with_security_span = require("../../../../utils/with_security_span");
var _enrichments = require("./enrichments");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// search_after through documents and re-index using bulk endpoint.
const searchAfterAndBulkCreate = async ({
  buildReasonMessage,
  bulkCreate,
  enrichment = _lodash.identity,
  eventsTelemetry,
  exceptionsList,
  filter,
  inputIndexPattern,
  listClient,
  pageSize,
  ruleExecutionLogger,
  services,
  sortOrder,
  trackTotalHits,
  tuple,
  wrapHits,
  runtimeMappings,
  primaryTimestamp,
  secondaryTimestamp,
  additionalFilters
}) => {
  return (0, _with_security_span.withSecuritySpan)('searchAfterAndBulkCreate', async () => {
    let toReturn = (0, _utils.createSearchAfterReturnType)();
    let searchingIteration = 0;

    // sortId tells us where to start our next consecutive search_after query
    let sortIds;
    let hasSortId = true; // default to true so we execute the search on initial run

    if (tuple == null || tuple.to == null || tuple.from == null) {
      ruleExecutionLogger.error(`missing run options fields: ${!tuple.to ? '"tuple.to"' : ''}, ${!tuple.from ? '"tuple.from"' : ''}`);
      return (0, _utils.createSearchAfterReturnType)({
        success: false,
        errors: ['malformed date tuple']
      });
    }
    while (toReturn.createdSignalsCount <= tuple.maxSignals) {
      const cycleNum = `cycle ${searchingIteration++}`;
      try {
        let mergedSearchResults = (0, _utils.createSearchResultReturnType)();
        ruleExecutionLogger.debug(`[${cycleNum}] Searching events${sortIds ? ` after cursor ${JSON.stringify(sortIds)}` : ''} in index pattern "${inputIndexPattern}"`);
        if (hasSortId) {
          var _searchResult$hits$hi;
          const {
            searchResult,
            searchDuration,
            searchErrors
          } = await (0, _single_search_after.singleSearchAfter)({
            searchAfterSortIds: sortIds,
            index: inputIndexPattern,
            runtimeMappings,
            from: tuple.from.toISOString(),
            to: tuple.to.toISOString(),
            services,
            ruleExecutionLogger,
            filter,
            pageSize: Math.ceil(Math.min(tuple.maxSignals, pageSize)),
            primaryTimestamp,
            secondaryTimestamp,
            trackTotalHits,
            sortOrder,
            additionalFilters
          });
          mergedSearchResults = (0, _utils.mergeSearchResults)([mergedSearchResults, searchResult]);
          toReturn = (0, _utils.mergeReturns)([toReturn, (0, _utils.createSearchAfterReturnTypeFromResponse)({
            searchResult: mergedSearchResults,
            primaryTimestamp
          }), (0, _utils.createSearchAfterReturnType)({
            searchAfterTimes: [searchDuration],
            errors: searchErrors
          })]);

          // determine if there are any candidate signals to be processed
          const totalHits = (0, _utils.getTotalHitsValue)(mergedSearchResults.hits.total);
          const lastSortIds = (0, _utils.getSafeSortIds)((_searchResult$hits$hi = searchResult.hits.hits[searchResult.hits.hits.length - 1]) === null || _searchResult$hits$hi === void 0 ? void 0 : _searchResult$hits$hi.sort);
          if (totalHits === 0 || mergedSearchResults.hits.hits.length === 0) {
            ruleExecutionLogger.debug(`[${cycleNum}] Found 0 events ${sortIds ? ` after cursor ${JSON.stringify(sortIds)}` : ''}`);
            break;
          } else {
            ruleExecutionLogger.debug(`[${cycleNum}] Found ${mergedSearchResults.hits.hits.length} of total ${totalHits} events${sortIds ? ` after cursor ${JSON.stringify(sortIds)}` : ''}, last cursor ${JSON.stringify(lastSortIds)}`);
          }
          if (lastSortIds != null && lastSortIds.length !== 0) {
            sortIds = lastSortIds;
            hasSortId = true;
          } else {
            hasSortId = false;
          }
        }

        // filter out the search results that match with the values found in the list.
        // the resulting set are signals to be indexed, given they are not duplicates
        // of signals already present in the signals index.
        const [includedEvents, _] = await (0, _filter_events_against_list.filterEventsAgainstList)({
          listClient,
          exceptionsList,
          ruleExecutionLogger,
          events: mergedSearchResults.hits.hits
        });

        // only bulk create if there are filteredEvents leftover
        // if there isn't anything after going through the value list filter
        // skip the call to bulk create and proceed to the next search_after,
        // if there is a sort id to continue the search_after with.
        if (includedEvents.length !== 0) {
          const enrichedEvents = await enrichment(includedEvents);
          const wrappedDocs = wrapHits(enrichedEvents, buildReasonMessage);
          const bulkCreateResult = await bulkCreate(wrappedDocs, tuple.maxSignals - toReturn.createdSignalsCount, (0, _enrichments.createEnrichEventsFunction)({
            services,
            logger: ruleExecutionLogger
          }));
          if (bulkCreateResult.alertsWereTruncated) {
            toReturn.warningMessages.push((0, _utils.getMaxSignalsWarning)());
            break;
          }
          (0, _utils.addToSearchAfterReturn)({
            current: toReturn,
            next: bulkCreateResult
          });
          ruleExecutionLogger.debug(`[${cycleNum}] Created ${bulkCreateResult.createdItemsCount} alerts from ${enrichedEvents.length} events`);
          (0, _send_telemetry_events.sendAlertTelemetryEvents)(enrichedEvents, bulkCreateResult.createdItems, eventsTelemetry, ruleExecutionLogger);
        }
        if (!hasSortId) {
          ruleExecutionLogger.debug(`[${cycleNum}] Unable to fetch last event cursor`);
          break;
        }
      } catch (exc) {
        ruleExecutionLogger.error('Unable to extract/process events or create alerts', JSON.stringify(exc));
        return (0, _utils.mergeReturns)([toReturn, (0, _utils.createSearchAfterReturnType)({
          success: false,
          errors: [`${exc}`]
        })]);
      }
    }
    ruleExecutionLogger.debug(`Completed bulk indexing of ${toReturn.createdSignalsCount} alert`);
    return toReturn;
  });
};
exports.searchAfterAndBulkCreate = searchAfterAndBulkCreate;