"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.thresholdExecutor = void 0;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _get_filter = require("../utils/get_filter");
var _bulk_create_threshold_signals = require("./bulk_create_threshold_signals");
var _find_threshold_signals = require("./find_threshold_signals");
var _get_threshold_bucket_filters = require("./get_threshold_bucket_filters");
var _get_threshold_signal_history = require("./get_threshold_signal_history");
var _bulk_create_suppressed_threshold_alerts = require("./bulk_create_suppressed_threshold_alerts");
var _utils = require("../utils/utils");
var _with_security_span = require("../../../../utils/with_security_span");
var _build_signal_history = require("./build_signal_history");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const thresholdExecutor = async ({
  inputIndex,
  runtimeMappings,
  completeRule,
  tuple,
  ruleExecutionLogger,
  services,
  version,
  startedAt,
  state,
  bulkCreate,
  wrapHits,
  ruleDataClient,
  primaryTimestamp,
  secondaryTimestamp,
  aggregatableTimestampField,
  exceptionFilter,
  unprocessedExceptions,
  inputIndexFields,
  spaceId,
  runOpts,
  licensing
}) => {
  const result = (0, _utils.createSearchAfterReturnType)();
  const ruleParams = completeRule.ruleParams;
  return (0, _with_security_span.withSecuritySpan)('thresholdExecutor', async () => {
    const exceptionsWarning = (0, _utils.getUnprocessedExceptionsWarnings)(unprocessedExceptions);
    if (exceptionsWarning) {
      result.warningMessages.push(exceptionsWarning);
    }
    const license = await (0, _rxjs.firstValueFrom)(licensing.license$);
    const hasPlatinumLicense = license.hasAtLeast('platinum');

    // Get state or build initial state (on upgrade)
    const {
      signalHistory,
      searchErrors: previousSearchErrors
    } = state.initialized ? {
      signalHistory: state.signalHistory,
      searchErrors: []
    } : await (0, _get_threshold_signal_history.getThresholdSignalHistory)({
      from: tuple.from.toISOString(),
      to: tuple.to.toISOString(),
      frameworkRuleId: completeRule.alertId,
      bucketByFields: ruleParams.threshold.field,
      spaceId,
      ruleDataClient,
      esClient: services.scopedClusterClient.asCurrentUser
    });
    const validSignalHistory = (0, _utils2.getSignalHistory)(state, signalHistory, tuple);
    // Eliminate dupes
    const bucketFilters = await (0, _get_threshold_bucket_filters.getThresholdBucketFilters)({
      signalHistory: validSignalHistory,
      aggregatableTimestampField
    });

    // Combine dupe filter with other filters
    const esFilter = await (0, _get_filter.getFilter)({
      type: ruleParams.type,
      filters: ruleParams.filters ? ruleParams.filters.concat(bucketFilters) : bucketFilters,
      language: ruleParams.language,
      query: ruleParams.query,
      savedId: ruleParams.savedId,
      services,
      index: inputIndex,
      exceptionFilter,
      fields: inputIndexFields
    });

    // Look for new events over threshold
    const {
      buckets,
      searchErrors,
      searchDurations,
      warnings
    } = await (0, _find_threshold_signals.findThresholdSignals)({
      inputIndexPattern: inputIndex,
      from: tuple.from.toISOString(),
      to: tuple.to.toISOString(),
      maxSignals: tuple.maxSignals,
      services,
      ruleExecutionLogger,
      filter: esFilter,
      threshold: ruleParams.threshold,
      runtimeMappings,
      primaryTimestamp,
      secondaryTimestamp,
      aggregatableTimestampField
    });
    const alertSuppression = completeRule.ruleParams.alertSuppression;
    let createResult;
    let newSignalHistory;
    if (alertSuppression !== null && alertSuppression !== void 0 && alertSuppression.duration && hasPlatinumLicense) {
      const suppressedResults = await (0, _bulk_create_suppressed_threshold_alerts.bulkCreateSuppressedThresholdAlerts)({
        buckets,
        completeRule,
        services,
        inputIndexPattern: inputIndex,
        startedAt,
        from: tuple.from.toDate(),
        to: tuple.to.toDate(),
        ruleExecutionLogger,
        spaceId,
        runOpts
      });
      createResult = suppressedResults.bulkCreateResult;
      newSignalHistory = (0, _build_signal_history.buildThresholdSignalHistory)({
        alerts: suppressedResults.unsuppressedAlerts
      });
    } else {
      createResult = await (0, _bulk_create_threshold_signals.bulkCreateThresholdSignals)({
        buckets,
        completeRule,
        filter: esFilter,
        services,
        inputIndexPattern: inputIndex,
        signalsIndex: ruleParams.outputIndex,
        startedAt,
        from: tuple.from.toDate(),
        signalHistory: validSignalHistory,
        bulkCreate,
        wrapHits,
        ruleExecutionLogger
      });
      newSignalHistory = (0, _build_signal_history.buildThresholdSignalHistory)({
        alerts: (0, _utils2.transformBulkCreatedItemsToHits)(createResult.createdItems)
      });
    }
    (0, _utils.addToSearchAfterReturn)({
      current: result,
      next: {
        ...createResult,
        success: createResult.success && (0, _lodash.isEmpty)(searchErrors)
      }
    });
    result.errors.push(...previousSearchErrors);
    result.errors.push(...searchErrors);
    result.warningMessages.push(...warnings);
    result.searchAfterTimes = searchDurations;
    return {
      ...result,
      state: {
        ...state,
        initialized: true,
        signalHistory: {
          ...validSignalHistory,
          ...newSignalHistory
        }
      }
    };
  });
};
exports.thresholdExecutor = thresholdExecutor;