"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.stripNonEcsFields = void 0;
var _alertsAsDataUtils = require("@kbn/alerts-as-data-utils");
var _lodash = require("lodash");
var _is_valid_ip_type = require("./ecs_types_validators/is_valid_ip_type");
var _is_valid_date_type = require("./ecs_types_validators/is_valid_date_type");
var _is_valid_numeric_type = require("./ecs_types_validators/is_valid_numeric_type");
var _is_valid_boolean_type = require("./ecs_types_validators/is_valid_boolean_type");
var _is_valid_long_type = require("./ecs_types_validators/is_valid_long_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * type guard for object of SearchTypes
 */
const isSearchTypesRecord = value => {
  return (0, _lodash.isPlainObject)(value);
};

/**
 * retrieve all nested object fields from ecsFieldMap
 * field `agent.build.original` will be converted into
 * { agent: true, agent.build: true }
 */
const getEcsObjectFields = () => {
  const result = {};
  Object.entries(_alertsAsDataUtils.ecsFieldMap).forEach(([key, value]) => {
    const objects = key.split('.');
    // last item can be any of type, as precedent are objects
    objects.pop();
    objects.reduce((parentPath, itemKey) => {
      const fullPath = parentPath ? `${parentPath}.${itemKey}` : itemKey;
      if (!result[fullPath]) {
        result[fullPath] = true;
      }
      return fullPath;
    }, '');
  });
  return result;
};
const ecsObjectFields = getEcsObjectFields();

/**
 * checks if path is a valid Ecs object type (object or flattened)
 * geo_point also can be object
 */
const getIsEcsFieldObject = path => {
  const ecsField = _alertsAsDataUtils.ecsFieldMap[path];
  return ['object', 'flattened', 'geo_point'].includes(ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) || ecsObjectFields[path];
};

/**
 * checks if path is in Ecs mapping
 */
const getIsEcsField = path => {
  const ecsField = _alertsAsDataUtils.ecsFieldMap[path];
  const isEcsField = !!ecsField || ecsObjectFields[path];
  return isEcsField;
};

/**
 * if any of partial path in dotted notation is not an object in ECS mapping
 * it means the field itself is not valid as well
 * For example, 'agent.name.conflict' - if agent.name is keyword, so the whole path is invalid
 */
const validateDottedPathInEcsMappings = path => {
  let isValid = true;
  path.split('.').slice(0, -1) // exclude last path item, as we check only if all parent are objects
  .reduce((acc, key) => {
    const pathToValidate = [acc, key].filter(Boolean).join('.');
    const isEcsField = getIsEcsField(pathToValidate);
    const isEcsFieldObject = getIsEcsFieldObject(pathToValidate);

    // if field is in Ecs mapping and not object, the whole path is invalid
    if (isEcsField && !isEcsFieldObject) {
      isValid = false;
    }
    return pathToValidate;
  }, '');
  return isValid;
};

/**
 * check whether source field value is ECS compliant
 */
const computeIsEcsCompliant = (value, path) => {
  // if path consists of dot-notation, ensure each path within it is ECS compliant (object or flattened)
  if (path.includes('.') && !validateDottedPathInEcsMappings(path)) {
    return false;
  }
  const isEcsField = getIsEcsField(path);

  // if field is not present is ECS mapping, it's valid as doesn't have any conflicts with existing mapping
  if (!isEcsField) {
    return true;
  }
  const ecsField = _alertsAsDataUtils.ecsFieldMap[path];
  const isEcsFieldObject = getIsEcsFieldObject(path);

  // do not validate geo_point, since it's very complex type that can be string/array/object
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'geo_point') {
    return true;
  }

  // validate if value is a long type
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'long') {
    return (0, _is_valid_long_type.isValidLongType)(value);
  }

  // validate if value is a numeric type
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'float' || (ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'scaled_float') {
    return (0, _is_valid_numeric_type.isValidNumericType)(value);
  }

  // validate if value is a valid ip type
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'ip') {
    return (0, _is_valid_ip_type.isValidIpType)(value);
  }

  // validate if value is a valid date
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'date') {
    return (0, _is_valid_date_type.isValidDateType)(value);
  }

  // validate if value is a valid boolean
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'boolean') {
    return (0, _is_valid_boolean_type.isValidBooleanType)(value);
  }

  // if ECS mapping is JS object and source value also JS object then they are compliant
  // otherwise not
  return isEcsFieldObject ? (0, _lodash.isPlainObject)(value) : !(0, _lodash.isPlainObject)(value);
};
/**
 * strips alert source object from ECS non compliant fields
 */
const stripNonEcsFields = doc => {
  const result = (0, _lodash.cloneDeep)(doc);
  const removed = [];

  /**
   * traverses through object and deletes ECS non compliant fields
   * @param document - document to traverse
   * @param documentKey - document key in parent document, if exists
   * @param parent - parent of traversing document
   * @param parentPath - path of parent in initial source document
   */
  const traverseAndDeleteInObj = (document, documentKey, parent, parentPath) => {
    const fullPath = [parentPath, documentKey].filter(Boolean).join('.');
    // if document array, traverse through each item w/o changing documentKey, parent, parentPath
    if ((0, _lodash.isArray)(document) && document.length > 0) {
      document.slice().forEach(value => {
        traverseAndDeleteInObj(value, documentKey, parent, parentPath);
      });
      return;
    }
    if (parent && !computeIsEcsCompliant(document, fullPath)) {
      const documentReference = parent[documentKey];
      // if document reference in parent is array, remove only this item from array
      // e.g. a boolean mapped field with values ['not-boolean', 'true'] should strip 'not-boolean' and leave 'true'
      if ((0, _lodash.isArray)(documentReference)) {
        const indexToDelete = documentReference.findIndex(item => item === document);
        documentReference.splice(indexToDelete, 1);
        if (documentReference.length === 0) {
          delete parent[documentKey];
        }
      } else {
        delete parent[documentKey];
      }
      removed.push({
        key: fullPath,
        value: document
      });
      return;
    }
    if (isSearchTypesRecord(document)) {
      Object.entries(document).forEach(([key, value]) => {
        traverseAndDeleteInObj(value, key, document, fullPath);
      });
    }
  };
  traverseAndDeleteInObj(result, '');
  return {
    result,
    removed
  };
};
exports.stripNonEcsFields = stripNonEcsFields;