"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRuleObjectsHealthClient = void 0;
var _event_log_constants = require("../../event_log/event_log_constants");
var _filters = require("./filters");
var _health_stats_for_cluster = require("./aggregations/health_stats_for_cluster");
var _health_stats_for_space = require("./aggregations/health_stats_for_space");
var _fetch_rule_by_id = require("./fetch_rule_by_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Client for calculating health stats based on rule objects (saved objects of type "alert").
 */

const createRuleObjectsHealthClient = (rulesClient, internalSavedObjectsClient, logger) => {
  return {
    async calculateRuleHealth(args) {
      const rule = await (0, _fetch_rule_by_id.fetchRuleById)(rulesClient, args.rule_id);
      return {
        state_at_the_moment: {
          rule
        },
        debug: {}
      };
    },
    async calculateSpaceHealth(args) {
      const aggs = (0, _health_stats_for_space.getSpaceHealthAggregation)();
      const aggregations = await rulesClient.aggregate({
        options: {
          filter: _filters.DETECTION_RULES_FILTER // make sure to query only detection rules
        },

        aggs
      });
      return {
        state_at_the_moment: (0, _health_stats_for_space.normalizeSpaceHealthAggregationResult)(aggregations),
        debug: {
          rulesClient: {
            request: {
              aggs
            },
            response: {
              aggregations
            }
          }
        }
      };
    },
    async calculateClusterHealth(args) {
      const aggs = (0, _health_stats_for_cluster.getClusterHealthAggregation)();
      const response = await internalSavedObjectsClient.find({
        type: _event_log_constants.RULE_SAVED_OBJECT_TYPE,
        // query rules
        filter: _filters.DETECTION_RULES_FILTER,
        // make sure to query only detection rules
        namespaces: ['*'],
        // aggregate rules in all Kibana spaces
        perPage: 0,
        // don't return rules in the response, we only need aggs
        aggs
      });
      return {
        state_at_the_moment: (0, _health_stats_for_cluster.normalizeClusterHealthAggregationResult)(response.aggregations),
        debug: {
          savedObjectsClient: {
            request: {
              aggs
            },
            response
          }
        }
      };
    }
  };
};
exports.createRuleObjectsHealthClient = createRuleObjectsHealthClient;