"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.reviewRuleUpgradeRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _lodash = require("lodash");
var _prebuilt_rules = require("../../../../../../common/api/detection_engine/prebuilt_rules");
var _invariant = require("../../../../../../common/utils/invariant");
var _utils = require("../../../routes/utils");
var _calculate_rule_diff = require("../../logic/diff/calculate_rule_diff");
var _prebuilt_rule_assets_client = require("../../logic/rule_assets/prebuilt_rule_assets_client");
var _prebuilt_rule_objects_client = require("../../logic/rule_objects/prebuilt_rule_objects_client");
var _fetch_rule_versions_triad = require("../../logic/rule_versions/fetch_rule_versions_triad");
var _get_version_buckets = require("../../model/rule_versions/get_version_buckets");
var _rule_converters = require("../../../rule_management/normalization/rule_converters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const reviewRuleUpgradeRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _prebuilt_rules.REVIEW_RULE_UPGRADE_URL,
    options: {
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '1',
    validate: {}
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const ctx = await context.resolve(['core', 'alerting']);
      const soClient = ctx.core.savedObjects.client;
      const rulesClient = ctx.alerting.getRulesClient();
      const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(soClient);
      const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
      const ruleVersionsMap = await (0, _fetch_rule_versions_triad.fetchRuleVersionsTriad)({
        ruleAssetsClient,
        ruleObjectsClient
      });
      const {
        upgradeableRules
      } = (0, _get_version_buckets.getVersionBuckets)(ruleVersionsMap);
      const ruleDiffCalculationResults = upgradeableRules.map(({
        current
      }) => {
        const ruleVersions = ruleVersionsMap.get(current.rule_id);
        (0, _invariant.invariant)(ruleVersions != null, 'ruleVersions not found');
        return (0, _calculate_rule_diff.calculateRuleDiff)(ruleVersions);
      });
      const body = {
        stats: calculateRuleStats(ruleDiffCalculationResults),
        rules: calculateRuleInfos(ruleDiffCalculationResults)
      };
      return response.ok({
        body
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.reviewRuleUpgradeRoute = reviewRuleUpgradeRoute;
const calculateRuleStats = results => {
  const allTags = new Set(results.flatMap(result => {
    var _result$ruleVersions$, _result$ruleVersions$2;
    return (_result$ruleVersions$ = (_result$ruleVersions$2 = result.ruleVersions.input.current) === null || _result$ruleVersions$2 === void 0 ? void 0 : _result$ruleVersions$2.tags) !== null && _result$ruleVersions$ !== void 0 ? _result$ruleVersions$ : [];
  }));
  return {
    num_rules_to_upgrade_total: results.length,
    tags: [...allTags].sort((a, b) => a.localeCompare(b))
  };
};
const calculateRuleInfos = results => {
  return results.map(result => {
    const {
      ruleDiff,
      ruleVersions
    } = result;
    const installedCurrentVersion = ruleVersions.input.current;
    const targetVersion = ruleVersions.input.target;
    (0, _invariant.invariant)(installedCurrentVersion != null, 'installedCurrentVersion not found');
    (0, _invariant.invariant)(targetVersion != null, 'targetVersion not found');
    const targetRule = {
      ...(0, _rule_converters.convertPrebuiltRuleAssetToRuleResponse)(targetVersion),
      id: installedCurrentVersion.id,
      revision: installedCurrentVersion.revision + 1,
      created_at: installedCurrentVersion.created_at,
      created_by: installedCurrentVersion.created_by,
      updated_at: new Date().toISOString(),
      updated_by: installedCurrentVersion.updated_by
    };
    return {
      id: installedCurrentVersion.id,
      rule_id: installedCurrentVersion.rule_id,
      revision: installedCurrentVersion.revision,
      current_rule: installedCurrentVersion,
      target_rule: targetRule,
      diff: {
        fields: (0, _lodash.pickBy)(ruleDiff.fields, fieldDiff => fieldDiff.has_update || fieldDiff.has_conflict),
        has_conflict: ruleDiff.has_conflict
      }
    };
  });
};