"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateAlertStatusByQuery = exports.updateAlertStatusByIds = exports.getUserPrivilege = exports.getSignalIndex = exports.getHostMetadata = exports.getCaseIdsFromAlertId = exports.fetchQueryRuleRegistryAlerts = exports.fetchQueryAlerts = exports.createSignalIndex = exports.createHostUnIsolation = exports.createHostIsolation = exports.checkSignalIndex = void 0;
var _common = require("@kbn/cases-plugin/common");
var _constants = require("../../../../../common/constants");
var _constants2 = require("../../../../../common/endpoint/constants");
var _kibana = require("../../../../common/lib/kibana");
var _endpoint_isolation = require("../../../../common/lib/endpoint_isolation");
var _resolve_path_variables = require("../../../../common/utils/resolve_path_variables");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetch Alerts by providing a query
 *
 * @param query String to match a dsl
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
const fetchQueryAlerts = async ({
  query,
  signal
}) => {
  return _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_QUERY_SIGNALS_URL, {
    version: '2023-10-31',
    method: 'POST',
    body: JSON.stringify(query),
    signal
  });
};

/**
 * Fetch Alerts by providing a query
 *
 * @param query String to match a dsl
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
exports.fetchQueryAlerts = fetchQueryAlerts;
const fetchQueryRuleRegistryAlerts = async ({
  query,
  signal
}) => {
  return _kibana.KibanaServices.get().http.fetch(_constants.ALERTS_AS_DATA_FIND_URL, {
    method: 'POST',
    body: JSON.stringify(query),
    signal
  });
};

/**
 * Update alert status by query
 *
 * @param query of alerts to update
 * @param status to update to('open' / 'closed' / 'acknowledged')
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.fetchQueryRuleRegistryAlerts = fetchQueryRuleRegistryAlerts;
const updateAlertStatusByQuery = async ({
  query,
  status,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_SIGNALS_STATUS_URL, {
  version: '2023-10-31',
  method: 'POST',
  body: JSON.stringify({
    conflicts: 'proceed',
    status,
    query
  }),
  signal
});

/**
 * Update alert status by signalIds
 *
 * @param signalIds List of signal ids to update
 * @param status to update to('open' / 'closed' / 'acknowledged')
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.updateAlertStatusByQuery = updateAlertStatusByQuery;
const updateAlertStatusByIds = async ({
  signalIds,
  status,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_SIGNALS_STATUS_URL, {
  version: '2023-10-31',
  method: 'POST',
  body: JSON.stringify({
    status,
    signal_ids: signalIds
  }),
  signal
});

/**
 * Fetch Signal Index
 *
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.updateAlertStatusByIds = updateAlertStatusByIds;
const getSignalIndex = async ({
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_INDEX_URL, {
  version: '2023-10-31',
  method: 'GET',
  signal
});

/**
 * Check Signal Index
 *
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.getSignalIndex = getSignalIndex;
const checkSignalIndex = async ({
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_ALERTS_INDEX_URL, {
  version: '2023-10-31',
  method: 'GET',
  signal
});

/**
 * Get User Privileges
 *
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.checkSignalIndex = checkSignalIndex;
const getUserPrivilege = async ({
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_PRIVILEGES_URL, {
  version: '2023-10-31',
  method: 'GET',
  signal
});

/**
 * Create Signal Index if needed it
 *
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.getUserPrivilege = getUserPrivilege;
const createSignalIndex = async ({
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_INDEX_URL, {
  version: '2023-10-31',
  method: 'POST',
  signal
});

/**
 * Get Host Isolation index
 *
 * @param agent id
 * @param optional comment for the isolation action
 * @param optional case ids if associated with an alert on the host
 *
 * @throws An error if response is not OK
 */
exports.createSignalIndex = createSignalIndex;
const createHostIsolation = async ({
  endpointId,
  comment = '',
  caseIds
}) => (0, _endpoint_isolation.isolateHost)({
  endpoint_ids: [endpointId],
  comment,
  case_ids: caseIds
});

/**
 * Unisolate a host
 *
 * @param agent id
 * @param optional comment for the unisolation action
 * @param optional case ids if associated with an alert on the host
 *
 * @throws An error if response is not OK
 */
exports.createHostIsolation = createHostIsolation;
const createHostUnIsolation = async ({
  endpointId,
  comment = '',
  caseIds
}) => (0, _endpoint_isolation.unIsolateHost)({
  endpoint_ids: [endpointId],
  comment,
  case_ids: caseIds
});

/**
 * Get list of associated case ids from alert id
 *
 * @param alert id
 */
exports.createHostUnIsolation = createHostUnIsolation;
const getCaseIdsFromAlertId = async ({
  alertId,
  owner
}) => _kibana.KibanaServices.get().http.fetch((0, _common.getCasesFromAlertsUrl)(alertId), {
  method: 'get',
  query: {
    ...(owner.length > 0 ? {
      owner
    } : {})
  }
});

/**
 * Get Host metadata
 *
 * @param host id
 */
exports.getCaseIdsFromAlertId = getCaseIdsFromAlertId;
const getHostMetadata = async ({
  agentId,
  signal
}) => _kibana.KibanaServices.get().http.fetch((0, _resolve_path_variables.resolvePathVariables)(_constants2.HOST_METADATA_GET_ROUTE, {
  id: agentId
}), {
  method: 'GET',
  signal,
  version: '2023-10-31'
});
exports.getHostMetadata = getHostMetadata;