"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getStreamObservable = exports.getPlaceholderObservable = void 0;
var _rxjs = require("rxjs");
var _eventstreamCodec = require("@smithy/eventstream-codec");
var _utilUtf = require("@smithy/util-utf8");
var _translations = require("../translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MIN_DELAY = 35;
/**
 * Returns an Observable that reads data from a ReadableStream and emits values representing the state of the data processing.
 *
 * @param reader - The ReadableStreamDefaultReader used to read data from the stream.
 * @param setLoading - A function to update the loading state.
 * @param isError - indicates whether the reader response is an error message or not
 * @returns {Observable<PromptObservableState>} An Observable that emits PromptObservableState
 */
const getStreamObservable = ({
  connectorTypeTitle,
  isError,
  reader,
  setLoading
}) => new _rxjs.Observable(observer => {
  observer.next({
    chunks: [],
    loading: true
  });
  const decoder = new TextDecoder();
  const chunks = [];
  // Initialize an empty string to store the OpenAI buffer.
  let openAIBuffer = '';

  // Initialize an empty Uint8Array to store the Bedrock concatenated buffer.
  let bedrockBuffer = new Uint8Array(0);
  function readOpenAI() {
    reader.read().then(({
      done,
      value
    }) => {
      try {
        if (done) {
          if (openAIBuffer) {
            chunks.push(getOpenAIChunks([openAIBuffer])[0]);
          }
          observer.next({
            chunks,
            message: chunks.join(''),
            loading: false
          });
          observer.complete();
          return;
        }
        const decoded = decoder.decode(value);
        let nextChunks;
        if (isError) {
          nextChunks = [`${_translations.API_ERROR}\n\n${JSON.parse(decoded).message}`];
        } else {
          const lines = decoded.split('\n');
          lines[0] = openAIBuffer + lines[0];
          openAIBuffer = lines.pop() || '';
          nextChunks = getOpenAIChunks(lines);
        }
        nextChunks.forEach(chunk => {
          chunks.push(chunk);
          observer.next({
            chunks,
            message: chunks.join(''),
            loading: true
          });
        });
      } catch (err) {
        observer.error(err);
        return;
      }
      readOpenAI();
    }).catch(err => {
      observer.error(err);
    });
  }
  function readBedrock() {
    reader.read().then(({
      done,
      value
    }) => {
      try {
        if (done) {
          observer.next({
            chunks,
            message: chunks.join(''),
            loading: false
          });
          observer.complete();
          return;
        }
        let content;
        if (isError) {
          content = `${_translations.API_ERROR}\n\n${JSON.parse(decoder.decode(value)).message}`;
          chunks.push(content);
          observer.next({
            chunks,
            message: chunks.join(''),
            loading: true
          });
        } else if (value != null) {
          const chunk = value;

          // Concatenate the current chunk to the existing buffer.
          bedrockBuffer = concatChunks(bedrockBuffer, chunk);
          // Get the length of the next message in the buffer.
          let messageLength = getMessageLength(bedrockBuffer);

          // Initialize an array to store fully formed message chunks.
          const buildChunks = [];
          // Process the buffer until no complete messages are left.
          while (bedrockBuffer.byteLength > 0 && bedrockBuffer.byteLength >= messageLength) {
            // Extract a chunk of the specified length from the buffer.
            const extractedChunk = bedrockBuffer.slice(0, messageLength);
            // Add the extracted chunk to the array of fully formed message chunks.
            buildChunks.push(extractedChunk);
            // Remove the processed chunk from the buffer.
            bedrockBuffer = bedrockBuffer.slice(messageLength);
            // Get the length of the next message in the updated buffer.
            messageLength = getMessageLength(bedrockBuffer);
          }
          const awsDecoder = new _eventstreamCodec.EventStreamCodec(_utilUtf.toUtf8, _utilUtf.fromUtf8);
          // Decode and parse each message chunk, extracting the 'completion' property.
          buildChunks.forEach(bChunk => {
            const event = awsDecoder.decode(bChunk);
            const body = JSON.parse(Buffer.from(JSON.parse(decoder.decode(event.body)).bytes, 'base64').toString());
            content = body.completion;
            chunks.push(content);
            observer.next({
              chunks,
              message: chunks.join(''),
              loading: true
            });
          });
        }
      } catch (err) {
        observer.error(err);
        return;
      }
      readBedrock();
    }).catch(err => {
      observer.error(err);
    });
  }
  // this should never actually happen
  function badConnector() {
    observer.next({
      chunks: [`Invalid connector type - ${connectorTypeTitle} is not a supported GenAI connector.`],
      message: `Invalid connector type - ${connectorTypeTitle} is not a supported GenAI connector.`,
      loading: false
    });
    observer.complete();
  }
  if (connectorTypeTitle === 'Amazon Bedrock') readBedrock();else if (connectorTypeTitle === 'OpenAI') readOpenAI();else badConnector();
  return () => {
    reader.cancel();
  };
}).pipe(
// append a timestamp of when each value was emitted
(0, _rxjs.timestamp)(),
// use the previous timestamp to calculate a target
// timestamp for emitting the next value
(0, _rxjs.scan)((acc, value) => {
  const lastTimestamp = acc.timestamp || 0;
  const emitAt = Math.max(lastTimestamp + MIN_DELAY, value.timestamp);
  return {
    timestamp: emitAt,
    value: value.value
  };
}),
// add the delay based on the elapsed time
// using concatMap(of(value).pipe(delay(50))
// leads to browser issues because timers
// are throttled when the tab is not active
(0, _rxjs.concatMap)(value => {
  const now = Date.now();
  const delayFor = value.timestamp - now;
  if (delayFor <= 0) {
    return (0, _rxjs.of)(value.value);
  }
  return (0, _rxjs.of)(value.value).pipe((0, _rxjs.delay)(delayFor));
}),
// set loading to false when the observable completes or errors out
(0, _rxjs.finalize)(() => setLoading(false)));

/**
 * Parses an OpenAI response from a string.
 * @param lines
 * @returns {string[]} - Parsed string array from the OpenAI response.
 */
exports.getStreamObservable = getStreamObservable;
const getOpenAIChunks = lines => {
  const nextChunk = lines.map(str => str.substring(6)).filter(str => !!str && str !== '[DONE]').map(line => {
    try {
      var _openaiResponse$choic, _openaiResponse$choic2;
      const openaiResponse = JSON.parse(line);
      return (_openaiResponse$choic = (_openaiResponse$choic2 = openaiResponse.choices[0]) === null || _openaiResponse$choic2 === void 0 ? void 0 : _openaiResponse$choic2.delta.content) !== null && _openaiResponse$choic !== void 0 ? _openaiResponse$choic : '';
    } catch (err) {
      return '';
    }
  });
  return nextChunk;
};

/**
 * Concatenates two Uint8Array buffers.
 *
 * @param {Uint8Array} a - First buffer.
 * @param {Uint8Array} b - Second buffer.
 * @returns {Uint8Array} - Concatenated buffer.
 */
function concatChunks(a, b) {
  const newBuffer = new Uint8Array(a.length + b.length);
  // Copy the contents of the first buffer to the new buffer.
  newBuffer.set(a);
  // Copy the contents of the second buffer to the new buffer starting from the end of the first buffer.
  newBuffer.set(b, a.length);
  return newBuffer;
}

/**
 * Gets the length of the next message from the buffer.
 *
 * @param {Uint8Array} buffer - Buffer containing the message.
 * @returns {number} - Length of the next message.
 */
function getMessageLength(buffer) {
  // If the buffer is empty, return 0.
  if (buffer.byteLength === 0) return 0;
  // Create a DataView to read the Uint32 value at the beginning of the buffer.
  const view = new DataView(buffer.buffer, buffer.byteOffset, buffer.byteLength);
  // Read and return the Uint32 value (message length).
  return view.getUint32(0, false);
}
const getPlaceholderObservable = () => new _rxjs.Observable();
exports.getPlaceholderObservable = getPlaceholderObservable;