"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEndpointAuthzInitialState = exports.calculateEndpointAuthz = void 0;
exports.hasEndpointExceptionsPrivilege = hasEndpointExceptionsPrivilege;
exports.hasKibanaPrivilege = hasKibanaPrivilege;
var _lodash = require("lodash");
var _constants = require("../response_actions/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Checks to see if a given Kibana privilege was granted.
 * Note that this only checks if the user has the privilege as part of their role. That
 * does not indicate that the user has the granted functionality behind that privilege
 * (ex. due to license level). To get an accurate representation of user's authorization
 * level, use `calculateEndpointAuthz()`
 *
 * @param fleetAuthz
 * @param privilege
 */
function hasKibanaPrivilege(fleetAuthz, privilege) {
  var _fleetAuthz$packagePr, _fleetAuthz$packagePr2, _fleetAuthz$packagePr3;
  return (_fleetAuthz$packagePr = (_fleetAuthz$packagePr2 = fleetAuthz.packagePrivileges) === null || _fleetAuthz$packagePr2 === void 0 ? void 0 : (_fleetAuthz$packagePr3 = _fleetAuthz$packagePr2.endpoint) === null || _fleetAuthz$packagePr3 === void 0 ? void 0 : _fleetAuthz$packagePr3.actions[privilege].executePackageAction) !== null && _fleetAuthz$packagePr !== void 0 ? _fleetAuthz$packagePr : false;
}
function hasEndpointExceptionsPrivilege(fleetAuthz, privilege) {
  var _fleetAuthz$endpointE, _fleetAuthz$endpointE2;
  return (_fleetAuthz$endpointE = (_fleetAuthz$endpointE2 = fleetAuthz.endpointExceptionsPrivileges) === null || _fleetAuthz$endpointE2 === void 0 ? void 0 : _fleetAuthz$endpointE2.actions[privilege]) !== null && _fleetAuthz$endpointE !== void 0 ? _fleetAuthz$endpointE : false;
}

/**
 * Used by both the server and the UI to generate the Authorization for access to Endpoint related
 * functionality
 *
 * @param licenseService
 * @param fleetAuthz
 * @param userRoles
 */
const calculateEndpointAuthz = (licenseService, fleetAuthz, userRoles = []) => {
  var _fleetAuthz$fleet$all;
  const isPlatinumPlusLicense = licenseService.isPlatinumPlus();
  const isEnterpriseLicense = licenseService.isEnterprise();
  const hasEndpointManagementAccess = userRoles.includes('superuser');
  const canWriteSecuritySolution = hasKibanaPrivilege(fleetAuthz, 'writeSecuritySolution');
  const canReadSecuritySolution = hasKibanaPrivilege(fleetAuthz, 'readSecuritySolution');
  const canWriteEndpointList = hasKibanaPrivilege(fleetAuthz, 'writeEndpointList');
  const canReadEndpointList = hasKibanaPrivilege(fleetAuthz, 'readEndpointList');
  const canWritePolicyManagement = hasKibanaPrivilege(fleetAuthz, 'writePolicyManagement');
  const canReadPolicyManagement = hasKibanaPrivilege(fleetAuthz, 'readPolicyManagement');
  const canWriteActionsLogManagement = hasKibanaPrivilege(fleetAuthz, 'writeActionsLogManagement');
  const canReadActionsLogManagement = hasKibanaPrivilege(fleetAuthz, 'readActionsLogManagement');
  const canIsolateHost = hasKibanaPrivilege(fleetAuthz, 'writeHostIsolation');
  const canUnIsolateHost = hasKibanaPrivilege(fleetAuthz, 'writeHostIsolationRelease');
  const canWriteProcessOperations = hasKibanaPrivilege(fleetAuthz, 'writeProcessOperations');
  const canWriteTrustedApplications = hasKibanaPrivilege(fleetAuthz, 'writeTrustedApplications');
  const canReadTrustedApplications = hasKibanaPrivilege(fleetAuthz, 'readTrustedApplications');
  const canWriteHostIsolationExceptions = hasKibanaPrivilege(fleetAuthz, 'writeHostIsolationExceptions');
  const canReadHostIsolationExceptions = hasKibanaPrivilege(fleetAuthz, 'readHostIsolationExceptions');
  const canAccessHostIsolationExceptions = hasKibanaPrivilege(fleetAuthz, 'accessHostIsolationExceptions');
  const canDeleteHostIsolationExceptions = hasKibanaPrivilege(fleetAuthz, 'deleteHostIsolationExceptions');
  const canWriteBlocklist = hasKibanaPrivilege(fleetAuthz, 'writeBlocklist');
  const canReadBlocklist = hasKibanaPrivilege(fleetAuthz, 'readBlocklist');
  const canWriteEventFilters = hasKibanaPrivilege(fleetAuthz, 'writeEventFilters');
  const canReadEventFilters = hasKibanaPrivilege(fleetAuthz, 'readEventFilters');
  const canWriteFileOperations = hasKibanaPrivilege(fleetAuthz, 'writeFileOperations');
  const canWriteExecuteOperations = hasKibanaPrivilege(fleetAuthz, 'writeExecuteOperations');
  const canReadEndpointExceptions = hasEndpointExceptionsPrivilege(fleetAuthz, 'showEndpointExceptions');
  const canWriteEndpointExceptions = hasEndpointExceptionsPrivilege(fleetAuthz, 'crudEndpointExceptions');
  const authz = {
    canWriteSecuritySolution,
    canReadSecuritySolution,
    canAccessFleet: (_fleetAuthz$fleet$all = fleetAuthz === null || fleetAuthz === void 0 ? void 0 : fleetAuthz.fleet.all) !== null && _fleetAuthz$fleet$all !== void 0 ? _fleetAuthz$fleet$all : false,
    canAccessEndpointManagement: hasEndpointManagementAccess,
    // TODO: is this one deprecated? it is the only place we need to check for superuser.
    canCreateArtifactsByPolicy: isPlatinumPlusLicense,
    canWriteEndpointList,
    canReadEndpointList,
    canWritePolicyManagement,
    canReadPolicyManagement,
    canWriteActionsLogManagement,
    canReadActionsLogManagement: canReadActionsLogManagement && isEnterpriseLicense,
    canAccessEndpointActionsLogManagement: canReadActionsLogManagement && isPlatinumPlusLicense,
    // ---------------------------------------------------------
    // Response Actions
    // ---------------------------------------------------------
    canIsolateHost: canIsolateHost && isPlatinumPlusLicense,
    canUnIsolateHost,
    canKillProcess: canWriteProcessOperations && isEnterpriseLicense,
    canSuspendProcess: canWriteProcessOperations && isEnterpriseLicense,
    canGetRunningProcesses: canWriteProcessOperations && isEnterpriseLicense,
    canAccessResponseConsole: false,
    // set further below
    canWriteExecuteOperations: canWriteExecuteOperations && isEnterpriseLicense,
    canWriteFileOperations: canWriteFileOperations && isEnterpriseLicense,
    // ---------------------------------------------------------
    // artifacts
    // ---------------------------------------------------------
    canWriteTrustedApplications,
    canReadTrustedApplications,
    canWriteHostIsolationExceptions: canWriteHostIsolationExceptions && isPlatinumPlusLicense,
    canAccessHostIsolationExceptions: canAccessHostIsolationExceptions && isPlatinumPlusLicense,
    canReadHostIsolationExceptions,
    canDeleteHostIsolationExceptions,
    canWriteBlocklist,
    canReadBlocklist,
    canWriteEventFilters,
    canReadEventFilters,
    canReadEndpointExceptions,
    canWriteEndpointExceptions
  };

  // Response console is only accessible when license is Enterprise and user has access to any
  // of the response actions except `release`. Sole access to `release` is something
  // that is supported for a user in a license downgrade scenario, and in that case, we don't want
  // to allow access to Response Console.
  authz.canAccessResponseConsole = isEnterpriseLicense && Object.values((0, _lodash.omit)(_constants.RESPONSE_CONSOLE_ACTION_COMMANDS_TO_REQUIRED_AUTHZ, 'release')).some(responseActionAuthzKey => {
    return authz[responseActionAuthzKey];
  });
  return authz;
};
exports.calculateEndpointAuthz = calculateEndpointAuthz;
const getEndpointAuthzInitialState = () => {
  return {
    canWriteSecuritySolution: false,
    canReadSecuritySolution: false,
    canAccessFleet: false,
    canAccessEndpointActionsLogManagement: false,
    canAccessEndpointManagement: false,
    canCreateArtifactsByPolicy: false,
    canWriteEndpointList: false,
    canReadEndpointList: false,
    canWritePolicyManagement: false,
    canReadPolicyManagement: false,
    canWriteActionsLogManagement: false,
    canReadActionsLogManagement: false,
    canIsolateHost: false,
    canUnIsolateHost: false,
    canKillProcess: false,
    canSuspendProcess: false,
    canGetRunningProcesses: false,
    canAccessResponseConsole: false,
    canWriteFileOperations: false,
    canWriteExecuteOperations: false,
    canWriteTrustedApplications: false,
    canReadTrustedApplications: false,
    canWriteHostIsolationExceptions: false,
    canAccessHostIsolationExceptions: false,
    canReadHostIsolationExceptions: false,
    canDeleteHostIsolationExceptions: false,
    canWriteBlocklist: false,
    canReadBlocklist: false,
    canWriteEventFilters: false,
    canReadEventFilters: false,
    canReadEndpointExceptions: false,
    canWriteEndpointExceptions: false
  };
};
exports.getEndpointAuthzInitialState = getEndpointAuthzInitialState;