"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexFleetAgentForHost = exports.deleteIndexedFleetAgents = exports.buildFleetAgentBulkCreateOperations = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _usage_tracker = require("./usage_tracker");
var _fleet_agent_generator = require("../data_generators/fleet_agent_generator");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultFleetAgentGenerator = new _fleet_agent_generator.FleetAgentGenerator();
/**
 * Indexes a Fleet Agent
 * (NOTE: ensure that fleet is setup first before calling this loading function)
 *
 * @param esClient
 * @param kbnClient
 * @param endpointHost
 * @param agentPolicyId
 * @param [kibanaVersion]
 * @param [fleetAgentGenerator]
 */
const indexFleetAgentForHost = _usage_tracker.usageTracker.track('indexFleetAgentForHost', async (esClient, kbnClient, endpointHost, agentPolicyId, kibanaVersion = '8.0.0', fleetAgentGenerator = defaultFleetAgentGenerator) => {
  const agentDoc = generateFleetAgentEsHitForEndpointHost(endpointHost, agentPolicyId, kibanaVersion, fleetAgentGenerator);
  await esClient.index({
    index: agentDoc._index,
    id: agentDoc._id,
    body: agentDoc._source,
    op_type: 'create',
    refresh: 'wait_for'
  }).catch(_utils.wrapErrorAndRejectPromise);
  return {
    fleetAgentsIndex: agentDoc._index,
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    agents: [agentDoc._source]
  };
});
exports.indexFleetAgentForHost = indexFleetAgentForHost;
const generateFleetAgentEsHitForEndpointHost = (endpointHost, agentPolicyId, kibanaVersion = '8.0.0', fleetAgentGenerator = defaultFleetAgentGenerator) => {
  return fleetAgentGenerator.generateEsHit({
    _id: endpointHost.agent.id,
    _source: {
      agent: {
        id: endpointHost.agent.id,
        version: endpointHost.agent.version
      },
      local_metadata: {
        elastic: {
          agent: {
            id: endpointHost.agent.id,
            version: kibanaVersion
          }
        },
        host: {
          ...endpointHost.host
        },
        os: {
          ...endpointHost.host.os
        }
      },
      policy_id: agentPolicyId
    }
  });
};
/**
 * Creates an array of ES records with Fleet Agents that are associated with the provided set of Endpoint Agents.
 * Array can be used with the `bulk()` API's `operations` option.
 * @param endpoints
 * @param agentPolicyId
 * @param kibanaVersion
 * @param fleetAgentGenerator
 */
const buildFleetAgentBulkCreateOperations = ({
  endpoints,
  agentPolicyId,
  kibanaVersion = '8.0.0',
  fleetAgentGenerator = defaultFleetAgentGenerator
}) => {
  const response = {
    operations: [],
    agents: [],
    fleetAgentsIndex: _common.AGENTS_INDEX
  };
  for (const endpointHost of endpoints) {
    const agentDoc = generateFleetAgentEsHitForEndpointHost(endpointHost, agentPolicyId, kibanaVersion, fleetAgentGenerator);
    response.operations.push({
      create: {
        _index: agentDoc._index,
        _id: agentDoc._id
      }
    }, agentDoc._source);
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    response.agents.push(agentDoc._source);
  }
  return response;
};
exports.buildFleetAgentBulkCreateOperations = buildFleetAgentBulkCreateOperations;
const deleteIndexedFleetAgents = async (esClient, indexedData) => {
  const response = {
    agents: undefined
  };
  if (indexedData.agents.length) {
    response.agents = await esClient.deleteByQuery({
      index: `${indexedData.fleetAgentsIndex}-*`,
      wait_for_completion: true,
      body: {
        query: {
          bool: {
            filter: [{
              terms: {
                'local_metadata.elastic.agent.id': indexedData.agents.map(agent => agent.local_metadata.elastic.agent.id)
              }
            }]
          }
        }
      }
    }).catch(_utils.wrapErrorAndRejectPromise);
  }
  return response;
};
exports.deleteIndexedFleetAgents = deleteIndexedFleetAgents;