"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.labsServiceFactory = void 0;
var _common = require("../../../common");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const clearLabsFromStorage = storage => {
  _common.projectIDs.forEach(projectID => storage.removeItem(projectID));

  // This is a redundancy, to catch any labs that may have been removed above.
  // We could consider gathering telemetry to see how often this happens, or this may be unnecessary.
  Object.keys(storage).filter(key => key.startsWith(_common.LABS_PROJECT_PREFIX)).forEach(key => storage.removeItem(key));
};
const labsServiceFactory = ({
  coreStart
}) => {
  const {
    uiSettings
  } = coreStart;
  const localStorage = window.localStorage;
  const sessionStorage = window.sessionStorage;
  const getProjects = (solutions = []) => _common.projectIDs.reduce((acc, id) => {
    const project = getProject(id);
    if (solutions.length === 0 || solutions.some(solution => project.solutions.includes(solution))) {
      acc[id] = project;
    }
    return acc;
  }, {});
  const getProject = id => {
    const project = _common.projects[id];
    const status = {
      session: (0, _types.isEnabledByStorageValue)(project, 'session', sessionStorage.getItem(id)),
      browser: (0, _types.isEnabledByStorageValue)(project, 'browser', localStorage.getItem(id)),
      kibana: (0, _types.isEnabledByStorageValue)(project, 'kibana', uiSettings.get(id, project.isActive))
    };
    return (0, _types.applyProjectStatus)(project, status);
  };
  const setProjectStatus = (name, env, status) => {
    switch (env) {
      case 'session':
        (0, _types.setStorageStatus)(sessionStorage, name, status);
        break;
      case 'browser':
        (0, _types.setStorageStatus)(localStorage, name, status);
        break;
      case 'kibana':
        (0, _types.setUISettingsStatus)(uiSettings, name, status);
        break;
    }
  };
  const reset = () => {
    clearLabsFromStorage(localStorage);
    clearLabsFromStorage(sessionStorage);
    _common.projectIDs.forEach(id => setProjectStatus(id, 'kibana', _common.projects[id].isActive));
  };
  const isProjectEnabled = id => getProject(id).status.isEnabled;
  return {
    getProjectIDs: _common.getProjectIDs,
    getProjects,
    getProject,
    isProjectEnabled,
    reset,
    setProjectStatus
  };
};
exports.labsServiceFactory = labsServiceFactory;