"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncReduxEmbeddable = void 0;
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _clean_redux_embeddable_state = require("./clean_redux_embeddable_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const syncReduxEmbeddable = ({
  store,
  settings,
  embeddable,
  replaceEmbeddableReduxInput,
  replaceEmbeddableReduxOutput
}) => {
  if (settings !== null && settings !== void 0 && settings.disableSync) {
    return;
  }
  let embeddableToReduxInProgress = false;
  let reduxToEmbeddableInProgress = false;
  const {
    isInputEqual: inputEqualityCheck,
    isOutputEqual: outputEqualityCheck
  } = settings !== null && settings !== void 0 ? settings : {};
  const inputEqual = (inputA, inputB) => inputEqualityCheck ? inputEqualityCheck(inputA, inputB) : (0, _fastDeepEqual.default)(inputA, inputB);
  const outputEqual = (outputA, outputB) => outputEqualityCheck ? outputEqualityCheck(outputA, outputB) : (0, _fastDeepEqual.default)(outputA, outputB);

  // when the redux store changes, diff, and push updates to the embeddable input or to the output.
  const unsubscribeFromStore = store.subscribe(() => {
    if (embeddableToReduxInProgress) return;
    reduxToEmbeddableInProgress = true;
    const reduxState = store.getState();
    if (!inputEqual(reduxState.explicitInput, embeddable.getExplicitInput())) {
      embeddable.updateInput(reduxState.explicitInput);
    }
    if (!outputEqual(reduxState.output, embeddable.getOutput())) {
      // updating output is usually not accessible from outside of the embeddable.
      // This redux sync utility is meant to be used from inside the embeddable, so we need to workaround the typescript error via casting.
      embeddable.updateOutput(reduxState.output);
    }
    reduxToEmbeddableInProgress = false;
  });

  // when the embeddable input changes, diff and dispatch to the redux store
  const inputSubscription = embeddable.getInput$().subscribe(() => {
    if (reduxToEmbeddableInProgress) return;
    embeddableToReduxInProgress = true;
    const {
      explicitInput: reduxExplicitInput
    } = store.getState();

    // store only explicit input in the store
    const embeddableExplictInput = embeddable.getExplicitInput();
    if (!inputEqual(reduxExplicitInput, embeddableExplictInput)) {
      replaceEmbeddableReduxInput((0, _clean_redux_embeddable_state.cleanInputForRedux)(embeddableExplictInput));
    }
    embeddableToReduxInProgress = false;
  });

  // when the embeddable output changes, diff and dispatch to the redux store
  const outputSubscription = embeddable.getOutput$().subscribe(embeddableOutput => {
    if (reduxToEmbeddableInProgress) return;
    embeddableToReduxInProgress = true;
    const reduxState = store.getState();
    if (!outputEqual(reduxState.output, embeddableOutput)) {
      replaceEmbeddableReduxOutput(embeddableOutput);
    }
    embeddableToReduxInProgress = false;
  });
  return () => {
    unsubscribeFromStore();
    inputSubscription.unsubscribe();
    outputSubscription.unsubscribe();
  };
};
exports.syncReduxEmbeddable = syncReduxEmbeddable;