"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UpdateSLO = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _lodash = require("lodash");
var _constants = require("../../../common/slo/constants");
var _slo_summary_pipeline_template = require("../../assets/ingest_templates/slo_summary_pipeline_template");
var _services = require("../../domain/services");
var _retry = require("../../utils/retry");
var _create_temp_summary = require("./summary_transform_generator/helpers/create_temp_summary");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class UpdateSLO {
  constructor(repository, transformManager, summaryTransformManager, esClient, logger, spaceId) {
    this.repository = repository;
    this.transformManager = transformManager;
    this.summaryTransformManager = summaryTransformManager;
    this.esClient = esClient;
    this.logger = logger;
    this.spaceId = spaceId;
  }
  async execute(sloId, params) {
    const originalSlo = await this.repository.findById(sloId);
    let updatedSlo = Object.assign({}, originalSlo, params, {
      groupBy: !!params.groupBy ? params.groupBy : originalSlo.groupBy
    });
    if ((0, _lodash.isEqual)(originalSlo, updatedSlo)) {
      return this.toResponse(originalSlo);
    }
    const fields = ['indicator', 'groupBy', 'timeWindow', 'objective', 'budgetingMethod', 'settings'];
    const requireRevisionBump = !(0, _lodash.isEqual)((0, _lodash.pick)(originalSlo, fields), (0, _lodash.pick)(updatedSlo, fields));
    updatedSlo = Object.assign(updatedSlo, {
      updatedAt: new Date(),
      revision: requireRevisionBump ? originalSlo.revision + 1 : originalSlo.revision
    });
    (0, _services.validateSLO)(updatedSlo);
    await this.repository.save(updatedSlo);
    if (!requireRevisionBump) {
      // At this point, we still need to update the summary pipeline to include the changes (name, desc, tags, ...) in the summary index
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.ingest.putPipeline((0, _slo_summary_pipeline_template.getSLOSummaryPipelineTemplate)(updatedSlo, this.spaceId)), {
        logger: this.logger
      });
      return this.toResponse(updatedSlo);
    }
    const updatedRollupTransformId = (0, _constants.getSLOTransformId)(updatedSlo.id, updatedSlo.revision);
    const updatedSummaryTransformId = (0, _constants.getSLOSummaryTransformId)(updatedSlo.id, updatedSlo.revision);
    try {
      await this.transformManager.install(updatedSlo);
      await this.transformManager.start(updatedRollupTransformId);
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.ingest.putPipeline((0, _slo_summary_pipeline_template.getSLOSummaryPipelineTemplate)(updatedSlo, this.spaceId)), {
        logger: this.logger
      });
      await this.summaryTransformManager.install(updatedSlo);
      await this.summaryTransformManager.start(updatedSummaryTransformId);
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.index({
        index: _constants.SLO_SUMMARY_TEMP_INDEX_NAME,
        id: `slo-${updatedSlo.id}`,
        document: (0, _create_temp_summary.createTempSummaryDocument)(updatedSlo, this.spaceId),
        refresh: true
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.error(`Cannot update the SLO [id: ${updatedSlo.id}, revision: ${updatedSlo.revision}]. Rolling back.`);

      // Restore the previous slo definition
      await this.repository.save(originalSlo);
      // delete the created resources for the updated slo
      await this.summaryTransformManager.stop(updatedSummaryTransformId);
      await this.summaryTransformManager.uninstall(updatedSummaryTransformId);
      await this.transformManager.stop(updatedRollupTransformId);
      await this.transformManager.uninstall(updatedRollupTransformId);
      await this.esClient.ingest.deletePipeline({
        id: (0, _constants.getSLOSummaryPipelineId)(updatedSlo.id, updatedSlo.revision)
      }, {
        ignore: [404]
      });
      throw err;
    }
    await this.deleteOriginalSLO(originalSlo);
    return this.toResponse(updatedSlo);
  }
  async deleteOriginalSLO(originalSlo) {
    try {
      const originalRollupTransformId = (0, _constants.getSLOTransformId)(originalSlo.id, originalSlo.revision);
      await this.transformManager.stop(originalRollupTransformId);
      await this.transformManager.uninstall(originalRollupTransformId);
      const originalSummaryTransformId = (0, _constants.getSLOSummaryTransformId)(originalSlo.id, originalSlo.revision);
      await this.summaryTransformManager.stop(originalSummaryTransformId);
      await this.summaryTransformManager.uninstall(originalSummaryTransformId);
      await this.esClient.ingest.deletePipeline({
        id: (0, _constants.getSLOSummaryPipelineId)(originalSlo.id, originalSlo.revision)
      }, {
        ignore: [404]
      });
    } catch (err) {
      // Any errors here should not prevent moving forward.
      // Worst case we keep rolling up data for the previous revision number.
    }
    await this.deleteRollupData(originalSlo.id, originalSlo.revision);
    await this.deleteSummaryData(originalSlo.id, originalSlo.revision);
  }
  async deleteRollupData(sloId, sloRevision) {
    await this.esClient.deleteByQuery({
      index: _constants.SLO_DESTINATION_INDEX_PATTERN,
      wait_for_completion: false,
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': sloId
            }
          }, {
            term: {
              'slo.revision': sloRevision
            }
          }]
        }
      }
    });
  }
  async deleteSummaryData(sloId, sloRevision) {
    await this.esClient.deleteByQuery({
      index: _constants.SLO_SUMMARY_DESTINATION_INDEX_PATTERN,
      refresh: true,
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': sloId
            }
          }, {
            term: {
              'slo.revision': sloRevision
            }
          }]
        }
      }
    });
  }
  toResponse(slo) {
    return _sloSchema.updateSLOResponseSchema.encode(slo);
  }
}
exports.UpdateSLO = UpdateSLO;